"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MathExpression = exports.Metric = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const alarm_1 = require("./alarm");
const metric_util_1 = require("./private/metric-util");
const statistic_1 = require("./private/statistic");
/**
 * A metric emitted by a service.
 *
 * The metric is a combination of a metric identifier (namespace, name and dimensions)
 * and an aggregation function (statistic, period and unit).
 *
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * Metric is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class Metric {
    /**
     *
     */
    constructor(props) {
        this.period = props.period || cdk.Duration.minutes(5);
        const periodSec = this.period.toSeconds();
        if (periodSec !== 1 && periodSec !== 5 && periodSec !== 10 && periodSec !== 30 && periodSec % 60 !== 0) {
            throw new Error(`'period' must be 1, 5, 10, 30, or a multiple of 60 seconds, received ${periodSec}`);
        }
        this.dimensions = props.dimensions;
        this.namespace = props.namespace;
        this.metricName = props.metricName;
        // Try parsing, this will throw if it's not a valid stat
        this.statistic = statistic_1.normalizeStatistic(props.statistic || 'Average');
        this.label = props.label;
        this.color = props.color;
        this.unit = props.unit;
        this.account = props.account;
        this.region = props.region;
    }
    /**
     * Grant permissions to the given identity to write metrics.
     *
     * @param grantee The IAM identity to give permissions to.
     */
    static grantPutMetricData(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['cloudwatch:PutMetricData'],
            resourceArns: ['*'],
        });
    }
    /**
     * Return a copy of Metric `with` properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.statistic === undefined || props.statistic === this.statistic)
            && (props.unit === undefined || props.unit === this.unit)
            && (props.account === undefined || props.account === this.account)
            && (props.region === undefined || props.region === this.region)
            // For these we're not going to do deep equality, misses some opportunity for optimization
            // but that's okay.
            && (props.dimensions === undefined)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new Metric({
            dimensions: ifUndefined(props.dimensions, this.dimensions),
            namespace: this.namespace,
            metricName: this.metricName,
            period: ifUndefined(props.period, this.period),
            statistic: ifUndefined(props.statistic, this.statistic),
            unit: ifUndefined(props.unit, this.unit),
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            account: ifUndefined(props.account, this.account),
            region: ifUndefined(props.region, this.region),
        });
    }
    /**
     * Attach the metric object to the given construct scope.
     *
     * Returns a Metric object that uses the account and region from the Stack
     * the given construct is defined in. If the metric is subsequently used
     * in a Dashboard or Alarm in a different Stack defined in a different
     * account or region, the appropriate 'region' and 'account' fields
     * will be added to it.
     *
     * If the scope we attach to is in an environment-agnostic stack,
     * nothing is done and the same Metric object is returned.
     */
    attachTo(scope) {
        const stack = cdk.Stack.of(scope);
        return this.with({
            region: cdk.Token.isUnresolved(stack.region) ? undefined : stack.region,
            account: cdk.Token.isUnresolved(stack.account) ? undefined : stack.account,
        });
    }
    /**
     * Inspect the details of the metric object.
     */
    toMetricConfig() {
        const dims = this.dimensionsAsList();
        return {
            metricStat: {
                dimensions: dims.length > 0 ? dims : undefined,
                namespace: this.namespace,
                metricName: this.metricName,
                period: this.period,
                statistic: this.statistic,
                unitFilter: this.unit,
                account: this.account,
                region: this.region,
            },
            renderingProperties: {
                color: this.color,
                label: this.label,
            },
        };
    }
    /**
     * Turn this metric object into an alarm configuration.
     */
    toAlarmConfig() {
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        const stat = statistic_1.parseStatistic(metricConfig.metricStat.statistic);
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: stat.type === 'simple' ? stat.statistic : undefined,
            extendedStatistic: stat.type === 'percentile' ? 'p' + stat.percentile : undefined,
            unit: this.unit,
        };
    }
    /**
     * Turn this metric object into a graph configuration.
     */
    toGraphConfig() {
        var _a, _b, _c, _d;
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            renderingProperties: {
                period: metricConfig.metricStat.period.toSeconds(),
                stat: metricConfig.metricStat.statistic,
                color: asString((_a = metricConfig.renderingProperties) === null || _a === void 0 ? void 0 : _a.color),
                label: asString((_b = metricConfig.renderingProperties) === null || _b === void 0 ? void 0 : _b.label),
            },
            // deprecated properties for backwards compatibility
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: metricConfig.metricStat.statistic,
            color: asString((_c = metricConfig.renderingProperties) === null || _c === void 0 ? void 0 : _c.color),
            label: asString((_d = metricConfig.renderingProperties) === null || _d === void 0 ? void 0 : _d.label),
            unit: this.unit,
        };
    }
    /**
     * Make a new Alarm for this metric.
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                statistic: props.statistic,
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    /**
     * Returns a string representation of an object.
     */
    toString() {
        return this.label || this.metricName;
    }
    /**
     * Return the dimensions of this Metric as a list of Dimension.
     */
    dimensionsAsList() {
        const dims = this.dimensions;
        if (dims === undefined) {
            return [];
        }
        const list = Object.keys(dims).sort().map(key => ({ name: key, value: dims[key] }));
        return list;
    }
}
exports.Metric = Metric;
function asString(x) {
    if (x === undefined) {
        return undefined;
    }
    if (typeof x !== 'string') {
        throw new Error(`Expected string, got ${x}`);
    }
    return x;
}
/**
 * A math expression built with metric(s) emitted by a service.
 *
 * The math expression is a combination of an expression (x+y) and metrics to apply expression on.
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * MathExpression is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class MathExpression {
    /**
     *
     */
    constructor(props) {
        this.period = props.period || cdk.Duration.minutes(5);
        this.expression = props.expression;
        this.usingMetrics = changeAllPeriods(props.usingMetrics, this.period);
        this.label = props.label;
        this.color = props.color;
        const invalidVariableNames = Object.keys(props.usingMetrics).filter(x => !validVariableName(x));
        if (invalidVariableNames.length > 0) {
            throw new Error(`Invalid variable names in expression: ${invalidVariableNames}. Must start with lowercase letter and only contain alphanumerics.`);
        }
        this.validateNoIdConflicts();
    }
    /**
     * Return a copy of Metric with properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new MathExpression({
            expression: this.expression,
            usingMetrics: this.usingMetrics,
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            period: ifUndefined(props.period, this.period),
        });
    }
    /**
     * Turn this metric object into an alarm configuration.
     */
    toAlarmConfig() {
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    /**
     * Turn this metric object into a graph configuration.
     */
    toGraphConfig() {
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    /**
     * Inspect the details of the metric object.
     */
    toMetricConfig() {
        return {
            mathExpression: {
                period: this.period.toSeconds(),
                expression: this.expression,
                usingMetrics: this.usingMetrics,
            },
            renderingProperties: {
                label: this.label,
                color: this.color,
            },
        };
    }
    /**
     * Make a new Alarm for this metric.
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    /**
     * Returns a string representation of an object.
     */
    toString() {
        return this.label || this.expression;
    }
    validateNoIdConflicts() {
        const seen = new Map();
        visit(this);
        function visit(metric) {
            metric_util_1.dispatchMetric(metric, {
                withStat() {
                    // Nothing
                },
                withExpression(expr) {
                    for (const [id, subMetric] of Object.entries(expr.usingMetrics)) {
                        const existing = seen.get(id);
                        if (existing && metric_util_1.metricKey(existing) !== metric_util_1.metricKey(subMetric)) {
                            throw new Error(`The ID '${id}' used for two metrics in the expression: '${subMetric}' and '${existing}'. Rename one.`);
                        }
                        seen.set(id, subMetric);
                        visit(subMetric);
                    }
                },
            });
        }
    }
}
exports.MathExpression = MathExpression;
const VALID_VARIABLE = new RegExp('^[a-z][a-zA-Z0-9_]*$');
function validVariableName(x) {
    return VALID_VARIABLE.test(x);
}
function ifUndefined(x, def) {
    if (x !== undefined) {
        return x;
    }
    return def;
}
/**
 * Change periods of all metrics in the map
 */
function changeAllPeriods(metrics, period) {
    const ret = {};
    for (const [id, metric] of Object.entries(metrics)) {
        ret[id] = changePeriod(metric, period);
    }
    return ret;
}
/**
 * Return a new metric object which is the same type as the input object, but with the period changed
 *
 * Relies on the fact that implementations of `IMetric` are also supposed to have
 * an implementation of `with` that accepts an argument called `period`. See `IModifiableMetric`.
 */
function changePeriod(metric, period) {
    if (isModifiableMetric(metric)) {
        return metric.with({ period });
    }
    throw new Error(`Metric object should also implement 'with': ${metric}`);
}
function isModifiableMetric(m) {
    return typeof m === 'object' && m !== null && !!m.with;
}
//# sourceMappingURL=data:application/json;base64,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