"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeAlarm = void 0;
const core_1 = require("@aws-cdk/core");
const alarm_base_1 = require("./alarm-base");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
/**
 * A Composite Alarm based on Alarm Rule.
 */
class CompositeAlarm extends alarm_base_1.AlarmBase {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: (_a = props.compositeAlarmName) !== null && _a !== void 0 ? _a : core_1.Lazy.stringValue({ produce: () => this.generateUniqueId() }),
        });
        if (props.alarmRule.renderAlarmRule().length > 10240) {
            throw new Error('Alarm Rule expression cannot be greater than 10240 characters, please reduce the conditions in the Alarm Rule');
        }
        this.alarmRule = props.alarmRule.renderAlarmRule();
        const alarm = new cloudwatch_generated_1.CfnCompositeAlarm(this, 'Resource', {
            alarmName: this.physicalName,
            alarmRule: this.alarmRule,
            alarmDescription: props.alarmDescription,
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.listValue({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.listValue({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.listValue({ produce: () => this.okActionArns }),
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing CloudWatch composite alarm provided an Name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param compositeAlarmName Composite Alarm Name.
     */
    static fromCompositeAlarmName(scope, id, compositeAlarmName) {
        const stack = core_1.Stack.of(scope);
        return this.fromCompositeAlarmArn(scope, id, stack.formatArn({
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: compositeAlarmName,
        }));
    }
    /**
     * Import an existing CloudWatch composite alarm provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param compositeAlarmArn Composite Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm/CompositeAlarmName).
     */
    static fromCompositeAlarmArn(scope, id, compositeAlarmArn) {
        class Import extends alarm_base_1.AlarmBase {
            constructor() {
                super(...arguments);
                this.alarmArn = compositeAlarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(compositeAlarmArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    generateUniqueId() {
        const name = this.node.uniqueId;
        if (name.length > 240) {
            return name.substring(0, 120) + name.substring(name.length - 120);
        }
        return name;
    }
}
exports.CompositeAlarm = CompositeAlarm;
//# sourceMappingURL=data:application/json;base64,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