"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const acm = require("@aws-cdk/aws-certificatemanager");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const route53 = require("@aws-cdk/aws-route53");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const opensearch_access_policy_1 = require("./opensearch-access-policy");
const opensearchservice_generated_1 = require("./opensearchservice.generated");
const perms = require("./perms");
/**
 * The minimum TLS version required for traffic to the domain.
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    /** Cipher suite TLS 1.2 */
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * Grant read permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read/write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read/write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Return the given named metric for this domain.
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensionsMap: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     */
    metricClusterIndexWritesBlocked(props) {
        return this.metric('ClusterIndexWritesBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * Provides an Amazon OpenSearch Service domain.
 */
class Domain extends DomainBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29;
        super(scope, id, {
            physicalName: props.domainName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_opensearchservice_DomainProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const defaultInstanceType = 'r5.large.search';
        const warmDefaultInstanceType = 'ultrawarm1.medium.search';
        const dedicatedMasterType = initializeInstanceType(defaultInstanceType, (_b = props.capacity) === null || _b === void 0 ? void 0 : _b.masterNodeInstanceType);
        const dedicatedMasterCount = (_d = (_c = props.capacity) === null || _c === void 0 ? void 0 : _c.masterNodes) !== null && _d !== void 0 ? _d : 0;
        const dedicatedMasterEnabled = cdk.Token.isUnresolved(dedicatedMasterCount) ? true : dedicatedMasterCount > 0;
        const instanceType = initializeInstanceType(defaultInstanceType, (_e = props.capacity) === null || _e === void 0 ? void 0 : _e.dataNodeInstanceType);
        const instanceCount = (_g = (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodes) !== null && _g !== void 0 ? _g : 1;
        const warmType = initializeInstanceType(warmDefaultInstanceType, (_h = props.capacity) === null || _h === void 0 ? void 0 : _h.warmInstanceType);
        const warmCount = (_k = (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.warmNodes) !== null && _k !== void 0 ? _k : 0;
        const warmEnabled = cdk.Token.isUnresolved(warmCount) ? true : warmCount > 0;
        const availabilityZoneCount = (_m = (_l = props.zoneAwareness) === null || _l === void 0 ? void 0 : _l.availabilityZoneCount) !== null && _m !== void 0 ? _m : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_p = (_o = props.zoneAwareness) === null || _o === void 0 ? void 0 : _o.enabled) !== null && _p !== void 0 ? _p : ((_q = props.zoneAwareness) === null || _q === void 0 ? void 0 : _q.availabilityZoneCount) != null;
        let securityGroups;
        let subnets;
        if (props.vpc) {
            subnets = selectSubnets(props.vpc, (_r = props.vpcSubnets) !== null && _r !== void 0 ? _r : [{ subnetType: ec2.SubnetType.PRIVATE }]);
            securityGroups = (_s = props.securityGroups) !== null && _s !== void 0 ? _s : [new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                    description: `Security group for domain ${this.node.id}`,
                })];
            this._connections = new ec2.Connections({ securityGroups });
        }
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (subnets && zoneAwarenessEnabled && new Set(subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        if ([dedicatedMasterType, instanceType, warmType].some(t => (!cdk.Token.isUnresolved(t) && !t.endsWith('.search')))) {
            throw new Error('Master, data and UltraWarm node instance types must end with ".search".');
        }
        if (!cdk.Token.isUnresolved(warmType) && !warmType.startsWith('ultrawarm')) {
            throw new Error('UltraWarm node instance type must start with "ultrawarm".');
        }
        const unsignedBasicAuthEnabled = (_t = props.useUnsignedBasicAuth) !== null && _t !== void 0 ? _t : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_u = props.encryptionAtRest) === null || _u === void 0 ? void 0 : _u.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.AnyPrincipal()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_v = props.fineGrainedAccessControl) === null || _v === void 0 ? void 0 : _v.masterUserArn;
        const masterUserNameProps = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_x = props.fineGrainedAccessControl) === null || _x === void 0 ? void 0 : _x.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_z = (_y = props.encryptionAtRest) === null || _y === void 0 ? void 0 : _y.enabled) !== null && _z !== void 0 ? _z : (((_0 = props.encryptionAtRest) === null || _0 === void 0 ? void 0 : _0.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_1 = props.nodeToNodeEncryption) !== null && _1 !== void 0 ? _1 : unsignedBasicAuthEnabled;
        const volumeSize = (_3 = (_2 = props.ebs) === null || _2 === void 0 ? void 0 : _2.volumeSize) !== null && _3 !== void 0 ? _3 : 10;
        const volumeType = (_5 = (_4 = props.ebs) === null || _4 === void 0 ? void 0 : _4.volumeType) !== null && _5 !== void 0 ? _5 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_7 = (_6 = props.ebs) === null || _6 === void 0 ? void 0 : _6.enabled) !== null && _7 !== void 0 ? _7 : true;
        const enforceHttps = (_8 = props.enforceHttps) !== null && _8 !== void 0 ? _8 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryDatanodeInstanceType(...instanceTypes) {
            return instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch/OpenSearch version, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/features-by-version.html
        const { versionNum: versionNum, isElasticsearchVersion } = parseVersion(props.version);
        if (isElasticsearchVersion) {
            if (versionNum <= 7.7 &&
                ![
                    1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                    6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                    7.7,
                ].includes(versionNum)) {
                throw new Error(`Unknown Elasticsearch version: ${versionNum}`);
            }
            if (versionNum < 5.1) {
                if ((_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.appLogEnabled) {
                    throw new Error('Error logs publishing requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if ((_10 = props.encryptionAtRest) === null || _10 === void 0 ? void 0 : _10.enabled) {
                    throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (props.cognitoDashboardsAuth != null) {
                    throw new Error('Cognito authentication for OpenSearch Dashboards requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                    throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.0) {
                if (props.nodeToNodeEncryption) {
                    throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.7) {
                if (unsignedBasicAuthEnabled) {
                    throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
                if (advancedSecurityEnabled) {
                    throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.8 && warmEnabled) {
                throw new Error('UltraWarm requires Elasticsearch version 6.8 or later or OpenSearch version 1.0 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/supported-instance-types.html
        if (isSomeInstanceType('i3', 'r6gd') && ebsEnabled) {
            throw new Error('I3 and R6GD instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && !(isElasticsearchVersion && versionNum <= 2.3)) {
            throw new Error('The t2.micro.search instance type supports only Elasticsearch versions 1.5 and 2.3.');
        }
        if (isSomeInstanceType('t2', 't3') && warmEnabled) {
            throw new Error('T2 and T3 instance types do not support UltraWarm storage.');
        }
        // Only R3, I3 and r6gd support instance storage, per
        // https://aws.amazon.com/opensearch-service/pricing/
        if (!ebsEnabled && !isEveryDatanodeInstanceType('r3', 'i3', 'r6gd')) {
            throw new Error('EBS volumes are required when using instance types other than r3, i3 or r6gd.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (((_11 = props.logging) === null || _11 === void 0 ? void 0 : _11.auditLogEnabled) && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        // Validate UltraWarm requirement for dedicated master nodes, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ultrawarm.html
        if (warmEnabled && !dedicatedMasterEnabled) {
            throw new Error('Dedicated master node is required when UltraWarm storage is enabled.');
        }
        let cfnVpcOptions;
        if (securityGroups && subnets) {
            cfnVpcOptions = {
                securityGroupIds: securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_13 = props.logging.slowSearchLogGroup) !== null && _13 !== void 0 ? _13 : new logs.LogGroup(this, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_14 = props.logging) === null || _14 === void 0 ? void 0 : _14.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_15 = props.logging.slowIndexLogGroup) !== null && _15 !== void 0 ? _15 : new logs.LogGroup(this, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_16 = props.logging) === null || _16 === void 0 ? void 0 : _16.appLogEnabled) {
            this.appLogGroup = (_17 = props.logging.appLogGroup) !== null && _17 !== void 0 ? _17 : new logs.LogGroup(this, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        if ((_18 = props.logging) === null || _18 === void 0 ? void 0 : _18.auditLogEnabled) {
            this.auditLogGroup = (_19 = props.logging.auditLogGroup) !== null && _19 !== void 0 ? _19 : new logs.LogGroup(this, 'AuditLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        const logPublishing = {};
        if (this.appLogGroup) {
            logPublishing.ES_APPLICATION_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.appLogGroup.logGroupArn,
            };
        }
        if (this.slowSearchLogGroup) {
            logPublishing.SEARCH_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowSearchLogGroup.logGroupArn,
            };
        }
        if (this.slowIndexLogGroup) {
            logPublishing.INDEX_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowIndexLogGroup.logGroupArn,
            };
        }
        if (this.auditLogGroup) {
            logPublishing.AUDIT_LOGS = {
                enabled: this.auditLogGroup != null,
                cloudWatchLogsLogGroupArn: (_20 = this.auditLogGroup) === null || _20 === void 0 ? void 0 : _20.logGroupArn,
            };
        }
        let customEndpointCertificate;
        if (props.customEndpoint) {
            if (props.customEndpoint.certificate) {
                customEndpointCertificate = props.customEndpoint.certificate;
            }
            else {
                customEndpointCertificate = new acm.Certificate(this, 'CustomEndpointCertificate', {
                    domainName: props.customEndpoint.domainName,
                    validation: props.customEndpoint.hostedZone ? acm.CertificateValidation.fromDns(props.customEndpoint.hostedZone) : undefined,
                });
            }
        }
        // Create the domain
        this.domain = new opensearchservice_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            engineVersion: props.version.version,
            clusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                warmEnabled: warmEnabled
                    ? warmEnabled
                    : undefined,
                warmCount: warmEnabled
                    ? warmCount
                    : undefined,
                warmType: warmEnabled
                    ? warmType
                    : undefined,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_21 = props.ebs) === null || _21 === void 0 ? void 0 : _21.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_23 = (_22 = props.encryptionAtRest) === null || _22 === void 0 ? void 0 : _22.kmsKey) === null || _23 === void 0 ? void 0 : _23.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: logPublishing,
            cognitoOptions: {
                enabled: props.cognitoDashboardsAuth != null,
                identityPoolId: (_24 = props.cognitoDashboardsAuth) === null || _24 === void 0 ? void 0 : _24.identityPoolId,
                roleArn: (_25 = props.cognitoDashboardsAuth) === null || _25 === void 0 ? void 0 : _25.role.roleArn,
                userPoolId: (_26 = props.cognitoDashboardsAuth) === null || _26 === void 0 ? void 0 : _26.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_27 = props.tlsSecurityPolicy) !== null && _27 !== void 0 ? _27 : TLSSecurityPolicy.TLS_1_0,
                ...props.customEndpoint && {
                    customEndpointEnabled: true,
                    customEndpoint: props.customEndpoint.domainName,
                    customEndpointCertificateArn: customEndpointCertificate.certificateArn,
                },
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_28 = this.masterUserPassword) === null || _28 === void 0 ? void 0 : _28.unsafeUnwrap(),
                    },
                }
                : undefined,
            advancedOptions: props.advancedOptions,
        });
        this.domain.applyRemovalPolicy(props.removalPolicy);
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            if (!cdk.Token.isUnresolved(props.domainName)) {
                // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/configuration-api.html#configuration-api-datatypes-domainname
                if (!props.domainName.match(/^[a-z0-9\-]+$/)) {
                    throw new Error(`Invalid domainName '${props.domainName}'. Valid characters are a-z (lowercase only), 0-9, and – (hyphen).`);
                }
                if (props.domainName.length < 3 || props.domainName.length > 28) {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must be between 3 and 28 characters`);
                }
                if (props.domainName[0] < 'a' || props.domainName[0] > 'z') {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must start with a lowercase letter`);
                }
            }
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainId = this.domain.getAtt('Id').toString();
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        if ((_29 = props.customEndpoint) === null || _29 === void 0 ? void 0 : _29.hostedZone) {
            new route53.CnameRecord(this, 'CnameRecord', {
                recordName: props.customEndpoint.domainName,
                zone: props.customEndpoint.hostedZone,
                domainName: this.domainEndpoint,
            });
        }
        this.encryptionAtRestOptions = props.encryptionAtRest;
        if (props.accessPolicies) {
            this.addAccessPolicies(...props.accessPolicies);
        }
        if (unsignedBasicAuthEnabled) {
            this.addAccessPolicies(unsignedAccessPolicy);
        }
    }
    /**
     * Creates a domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * Creates a domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     */
    static fromDomainAttributes(scope, id, attrs) {
        var _b;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_opensearchservice_DomainAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDomainAttributes);
            }
            throw error;
        }
        const { domainArn, domainEndpoint } = attrs;
        const domainName = (_b = cdk.Stack.of(scope).splitArn(domainArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName) !== null && _b !== void 0 ? _b : extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainId = domainName;
                this.domainEndpoint = domainEndpoint.replace(/^https?:\/\//, '');
            }
        };
    }
    /**
     * Manages network connections to the domain. This will throw an error in case the domain
     * is not placed inside a VPC.
     */
    get connections() {
        if (!this._connections) {
            throw new Error("Connections are only available on VPC enabled domains. Use the 'vpc' property to place a domain inside a VPC");
        }
        return this._connections;
    }
    /**
     * Add policy statements to the domain access policy
     */
    addAccessPolicies(...accessPolicyStatements) {
        var _b;
        if (accessPolicyStatements.length > 0) {
            if (!this.accessPolicy) {
                // Only create the custom resource after there are statements to set.
                this.accessPolicy = new opensearch_access_policy_1.OpenSearchAccessPolicy(this, 'AccessPolicy', {
                    domainName: this.domainName,
                    domainArn: this.domainArn,
                    accessPolicies: accessPolicyStatements,
                });
                if ((_b = this.encryptionAtRestOptions) === null || _b === void 0 ? void 0 : _b.kmsKey) {
                    // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/encryption-at-rest.html
                    // these permissions are documented as required during domain creation.
                    // while not strictly documented for updates as well, it stands to reason that an update
                    // operation might require these in case the cluster uses a kms key.
                    // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                    this.accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                        actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                        resources: [this.encryptionAtRestOptions.kmsKey.keyArn],
                        effect: iam.Effect.ALLOW,
                    }));
                }
            }
            else {
                this.accessPolicy.addAccessPolicies(...accessPolicyStatements);
            }
        }
    }
}
exports.Domain = Domain;
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "@aws-cdk/aws-opensearchservice.Domain", version: "1.153.1" };
/**
 * Given an Amazon OpenSearch Service domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Amazon OpenSearch Service domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an engine version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The engine version object
 */
function parseVersion(version) {
    const elasticsearchPrefix = 'Elasticsearch_';
    const openSearchPrefix = 'OpenSearch_';
    const isElasticsearchVersion = version.version.startsWith(elasticsearchPrefix);
    const versionStr = isElasticsearchVersion
        ? version.version.substring(elasticsearchPrefix.length)
        : version.version.substring(openSearchPrefix.length);
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return { versionNum: parseFloat(versionStr), isElasticsearchVersion };
        }
        else {
            return { versionNum: parseFloat(versionStr.substring(0, secondDot)), isElasticsearchVersion };
        }
    }
    catch (error) {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
function selectSubnets(vpc, vpcSubnets) {
    const selected = [];
    for (const selection of vpcSubnets) {
        selected.push(...vpc.selectSubnets(selection).subnets);
    }
    return selected;
}
/**
 * Initializes an instance type.
 *
 * @param defaultInstanceType Default instance type which is used if no instance type is provided
 * @param instanceType Instance type
 * @returns Instance type in lowercase (if provided) or default instance type
 */
function initializeInstanceType(defaultInstanceType, instanceType) {
    if (instanceType) {
        return cdk.Token.isUnresolved(instanceType) ? instanceType : instanceType.toLowerCase();
    }
    else {
        return defaultInstanceType;
    }
}
//# sourceMappingURL=data:application/json;base64,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