import { IRole } from '@aws-cdk/aws-iam';
import * as lambda from '@aws-cdk/aws-lambda';
import { Construct, Duration, IResource, Resource } from '@aws-cdk/core';
import { ICustomAttribute, RequiredAttributes } from './user-pool-attr';
/**
 * The different ways in which users of this pool can sign up or sign in.
 */
export interface SignInAliases {
    /**
     * Whether user is allowed to sign up or sign in with a username
     * @default true
     */
    readonly username?: boolean;
    /**
     * Whether a user is allowed to sign up or sign in with an email address
     * @default false
     */
    readonly email?: boolean;
    /**
     * Whether a user is allowed to sign up or sign in with a phone number
     * @default false
     */
    readonly phone?: boolean;
    /**
     * Whether a user is allowed to ign in with a secondary username, that can be set and modified after sign up.
     * Can only be used in conjunction with `USERNAME`.
     * @default false
     */
    readonly preferredUsername?: boolean;
}
/**
 * Attributes that can be automatically verified for users in a user pool.
 */
export interface AutoVerifiedAttrs {
    /**
     * Whether the email address of the user should be auto verified at sign up.
     * @default - true, if email is turned on for `signIn`. false, otherwise.
     */
    readonly email?: boolean;
    /**
     * Whether the phone number of the user should be auto verified at sign up.
     * @default - true, if phone is turned on for `signIn`. false, otherwise.
     */
    readonly phone?: boolean;
}
export interface UserPoolTriggers {
    /**
     * Creates an authentication challenge.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
     */
    readonly createAuthChallenge?: lambda.IFunction;
    /**
     * A custom Message AWS Lambda trigger.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
     */
    readonly customMessage?: lambda.IFunction;
    /**
     * Defines the authentication challenge.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
     */
    readonly defineAuthChallenge?: lambda.IFunction;
    /**
     * A post-authentication AWS Lambda trigger.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
     */
    readonly postAuthentication?: lambda.IFunction;
    /**
     * A post-confirmation AWS Lambda trigger.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
     */
    readonly postConfirmation?: lambda.IFunction;
    /**
     * A pre-authentication AWS Lambda trigger.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
     */
    readonly preAuthentication?: lambda.IFunction;
    /**
     * A pre-registration AWS Lambda trigger.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
     */
    readonly preSignUp?: lambda.IFunction;
    /**
     * A pre-token-generation AWS Lambda trigger.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
     */
    readonly preTokenGeneration?: lambda.IFunction;
    /**
     * A user-migration AWS Lambda trigger.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
     */
    readonly userMigration?: lambda.IFunction;
    /**
     * Verifies the authentication challenge response.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
     */
    readonly verifyAuthChallengeResponse?: lambda.IFunction;
    /**
     * Index signature
     */
    [trigger: string]: lambda.IFunction | undefined;
}
/**
 * The email verification style
 */
export declare enum VerificationEmailStyle {
    /** Verify email via code */
    CODE = "CONFIRM_WITH_CODE",
    /** Verify email via link */
    LINK = "CONFIRM_WITH_LINK"
}
/**
 * User pool configuration for user self sign up.
 */
export interface UserVerificationConfig {
    /**
     * The email subject template for the verification email sent to the user upon sign up.
     * See https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-templates.html to
     * learn more about message templates.
     * @default 'Verify your new account'
     */
    readonly emailSubject?: string;
    /**
     * The email body template for the verification email sent to the user upon sign up.
     * See https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-templates.html to
     * learn more about message templates.
     * @default 'Hello {username}, Your verification code is {####}'
     */
    readonly emailBody?: string;
    /**
     * Emails can be verified either using a code or a link.
     * Learn more at https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-email-verification-message-customization.html
     * @default VerificationEmailStyle.CODE
     */
    readonly emailStyle?: VerificationEmailStyle;
    /**
     * The message template for the verification SMS sent to the user upon sign up.
     * See https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-message-templates.html to
     * learn more about message templates.
     * @default 'The verification code to your new account is {####}'
     */
    readonly smsMessage?: string;
}
/**
 * User pool configuration when administrators sign users up.
 */
export interface UserInvitationConfig {
    /**
     * The template to the email subject that is sent to the user when an administrator signs them up to the user pool.
     * @default 'Your temporary password'
     */
    readonly emailSubject?: string;
    /**
     * The template to the email body that is sent to the user when an administrator signs them up to the user pool.
     * @default 'Your username is {username} and temporary password is {####}.'
     */
    readonly emailBody?: string;
    /**
     * The template to the SMS message that is sent to the user when an administrator signs them up to the user pool.
     * @default 'Your username is {username} and temporary password is {####}'
     */
    readonly smsMessage?: string;
}
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
export declare enum Mfa {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    OFF = "OFF",
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    OPTIONAL = "OPTIONAL",
    /** Users are required to configure an MFA, and have to use it to sign in. */
    REQUIRED = "ON"
}
/**
 * The different ways in which a user pool can obtain their MFA token for sign in.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
export interface MfaSecondFactor {
    /**
     * The MFA token is sent to the user via SMS to their verified phone numbers
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa-sms-text-message.html
     * @default true
     */
    readonly sms: boolean;
    /**
     * The MFA token is a time-based one time password that is generated by a hardware or software token
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa-totp.html
     * @default false
     */
    readonly otp: boolean;
}
/**
 * Password policy for User Pools.
 */
export interface PasswordPolicy {
    /**
     * The length of time the temporary password generated by an admin is valid.
     * This must be provided as whole days, like Duration.days(3) or Duration.hours(48).
     * Fractional days, such as Duration.hours(20), will generate an error.
     * @default Duration.days(7)
     */
    readonly tempPasswordValidity?: Duration;
    /**
     * Minimum length required for a user's password.
     * @default 8
     */
    readonly minLength?: number;
    /**
     * Whether the user is required to have lowercase characters in their password.
     * @default true
     */
    readonly requireLowercase?: boolean;
    /**
     * Whether the user is required to have uppercase characters in their password.
     * @default true
     */
    readonly requireUppercase?: boolean;
    /**
     * Whether the user is required to have digits in their password.
     * @default true
     */
    readonly requireDigits?: boolean;
    /**
     * Whether the user is required to have symbols in their password.
     * @default true
     */
    readonly requireSymbols?: boolean;
}
/**
 * Email settings for the user pool.
 */
export interface EmailSettings {
    /**
     * The 'from' address on the emails received by the user.
     * @default noreply@verificationemail.com
     */
    readonly from?: string;
    /**
     * The 'replyTo' address on the emails received by the user as defined by IETF RFC-5322.
     * When set, most email clients recognize to change 'to' line to this address when a reply is drafted.
     * @default - Not set.
     */
    readonly replyTo?: string;
}
/**
 * Props for the UserPool construct
 */
export interface UserPoolProps {
    /**
     * Name of the user pool
     *
     * @default - automatically generated name by CloudFormation at deploy time
     */
    readonly userPoolName?: string;
    /**
     * Whether self sign up should be enabled. This can be further configured via the `selfSignUp` property.
     * @default false
     */
    readonly selfSignUpEnabled?: boolean;
    /**
     * Configuration around users signing themselves up to the user pool.
     * Enable or disable self sign-up via the `selfSignUpEnabled` property.
     * @default - see defaults in UserVerificationConfig
     */
    readonly userVerification?: UserVerificationConfig;
    /**
     * Configuration around admins signing up users into a user pool.
     * @default - see defaults in UserInvitationConfig
     */
    readonly userInvitation?: UserInvitationConfig;
    /**
     * The IAM role that Cognito will assume while sending SMS messages.
     * @default - a new IAM role is created
     */
    readonly smsRole?: IRole;
    /**
     * The 'ExternalId' that Cognito service must using when assuming the `smsRole`, if the role is restricted with an 'sts:ExternalId' conditional.
     * Learn more about ExternalId here - https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html
     *
     * This property will be ignored if `smsRole` is not specified.
     * @default - No external id will be configured
     */
    readonly smsRoleExternalId?: string;
    /**
     * Methods in which a user registers or signs in to a user pool.
     * Allows either username with aliases OR sign in with email, phone, or both.
     *
     * Read the sections on usernames and aliases to learn more -
     * https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html
     *
     * To match with 'Option 1' in the above link, with a verified email, this property should be set to
     * `{ username: true, email: true }`. To match with 'Option 2' in the above link with both a verified email and phone
     * number, this property should be set to `{ email: true, phone: true }`.
     *
     * @default { username: true }
     */
    readonly signInAliases?: SignInAliases;
    /**
     * Attributes which Cognito will look to verify automatically upon user sign up.
     * EMAIL and PHONE are the only available options.
     *
     * @default - If `signIn` include email and/or phone, they will be included in `autoVerifiedAttributes` by default.
     * If absent, no attributes will be auto-verified.
     */
    readonly autoVerify?: AutoVerifiedAttrs;
    /**
     * The set of attributes that are required for every user in the user pool.
     * Read more on attributes here - https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html
     *
     * @default - No attributes are required.
     */
    readonly requiredAttributes?: RequiredAttributes;
    /**
     * Define a set of custom attributes that can be configured for each user in the user pool.
     *
     * @default - No custom attributes.
     */
    readonly customAttributes?: {
        [key: string]: ICustomAttribute;
    };
    /**
     * Configure whether users of this user pool can or are required use MFA to sign in.
     *
     * @default Mfa.OFF
     */
    readonly mfa?: Mfa;
    /**
     * Configure the MFA types that users can use in this user pool. Ignored if `mfa` is set to `OFF`.
     *
     * @default - { sms: true, oneTimePassword: false }, if `mfa` is set to `OPTIONAL` or `REQUIRED`.
     * { sms: false, oneTimePassword: false }, otherwise
     */
    readonly mfaSecondFactor?: MfaSecondFactor;
    /**
     * Password policy for this user pool.
     * @default - see defaults on each property of PasswordPolicy.
     */
    readonly passwordPolicy?: PasswordPolicy;
    /**
     * Email settings for a user pool.
     * @default - see defaults on each property of EmailSettings.
     */
    readonly emailSettings?: EmailSettings;
    /**
     * Lambda functions to use for supported Cognito triggers.
     *
     * @default - No Lambda triggers.
     */
    readonly lambdaTriggers?: UserPoolTriggers;
}
/**
 * Represents a Cognito UserPool
 */
export interface IUserPool extends IResource {
    /**
     * The physical ID of this user pool resource
     * @attribute
     */
    readonly userPoolId: string;
    /**
     * The ARN of this user pool resource
     * @attribute
     */
    readonly userPoolArn: string;
}
/**
 * Define a Cognito User Pool
 */
export declare class UserPool extends Resource implements IUserPool {
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope: Construct, id: string, userPoolId: string): IUserPool;
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope: Construct, id: string, userPoolArn: string): IUserPool;
    /**
     * The physical ID of this user pool resource
     */
    readonly userPoolId: string;
    /**
     * The ARN of the user pool
     */
    readonly userPoolArn: string;
    /**
     * User pool provider name
     * @attribute
     */
    readonly userPoolProviderName: string;
    /**
     * User pool provider URL
     * @attribute
     */
    readonly userPoolProviderUrl: string;
    private triggers;
    constructor(scope: Construct, id: string, props?: UserPoolProps);
    /**
     * Attach 'Create Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addCreateAuthChallengeTrigger(fn: lambda.IFunction): void;
    /**
     * Attach 'Custom Message' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
     * @param fn the lambda function to attach
     */
    addCustomMessageTrigger(fn: lambda.IFunction): void;
    /**
     * Attach 'Define Auth Challenge' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
     * @param fn the lambda function to attach
     */
    addDefineAuthChallengeTrigger(fn: lambda.IFunction): void;
    /**
     * Attach 'Post Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
     * @param fn the lambda function to attach
     */
    addPostAuthenticationTrigger(fn: lambda.IFunction): void;
    /**
     * Attach 'Post Confirmation' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
     * @param fn the lambda function to attach
     */
    addPostConfirmationTrigger(fn: lambda.IFunction): void;
    /**
     * Attach 'Pre Authentication' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
     * @param fn the lambda function to attach
     */
    addPreAuthenticationTrigger(fn: lambda.IFunction): void;
    /**
     * Attach 'Pre Sign Up' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
     * @param fn the lambda function to attach
     */
    addPreSignUpTrigger(fn: lambda.IFunction): void;
    /**
     * Attach 'Pre Token Generation' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
     * @param fn the lambda function to attach
     */
    addPreTokenGenerationTrigger(fn: lambda.IFunction): void;
    /**
     * Attach 'User Migration' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
     * @param fn the lambda function to attach
     */
    addUserMigrationTrigger(fn: lambda.IFunction): void;
    /**
     * Attach 'Verify Auth Challenge Response' trigger
     * Grants access from cognito-idp.amazonaws.com to the lambda
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
     * @param fn the lambda function to attach
     */
    addVerifyAuthChallengeResponseTrigger(fn: lambda.IFunction): void;
    private addLambdaPermission;
    private signInConfiguration;
    private smsConfiguration;
    private mfaConfiguration;
    private configurePasswordPolicy;
    private schemaConfiguration;
}
