"""
Worker 配置管理
"""

from typing import Dict, List, Optional

from pydantic import Field
from pydantic_settings import BaseSettings

from mineru_flow.internal.config.config import artifacts_dir
from mineru_flow.internal.schema.state import Phase


class WorkerConfig(BaseSettings):
    """Worker 配置"""

    # 基本配置
    artifacts_dir: str = Field(default=artifacts_dir, description="Artifacts 目录")
    concurrency: int = Field(default=4, description="Worker 并发数")
    polling_interval_ms: int = Field(default=5000, description="轮询间隔（毫秒）")
    max_retry_attempts: int = Field(default=3, description="最大重试次数")

    # 处理阶段配置
    processing_phases: List[str] = Field(
        default=[Phase.PARSE, Phase.CHUNK, Phase.IMPORT], description="处理阶段列表"
    )

    # 阶段转换映射（现在在worker中直接定义，不再使用配置）
    phase_transitions: Dict[str, Optional[str]] = Field(
        default={
            Phase.PARSE: Phase.CHUNK,
            Phase.CHUNK: Phase.IMPORT,
            Phase.IMPORT: None,
        },
        description="阶段转换映射",
    )

    # 超时配置
    task_timeout_seconds: int = Field(
        default=3600, description="单个任务超时时间（秒）"
    )
    phase_timeout_seconds: int = Field(
        default=1800, description="单个阶段超时时间（秒）"
    )

    # 错误处理配置
    error_backoff_seconds: int = Field(default=5, description="错误后退避时间（秒）")
    max_consecutive_errors: int = Field(default=10, description="最大连续错误数")

    class Config:
        env_prefix = "WORKER_"
        case_sensitive = False


# 全局配置实例
worker_config = WorkerConfig()


def get_worker_config() -> WorkerConfig:
    """获取 Worker 配置"""
    return worker_config
