# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dted', 'dted.records']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.20,<2.0']

extras_require = \
{':python_version < "3.8"': ['cached-property>1.5', 'importlib-metadata>4.0']}

entry_points = \
{'console_scripts': ['dted = dted.__main__:main']}

setup_kwargs = {
    'name': 'dted',
    'version': '1.0.4',
    'description': 'Parser for DTED data.',
    'long_description': '# DTED Parser\n\nThis is a package written in pure python (with help from `numpy`) to parse\n  and investigate Digital Terrain Elevation Data (DTED) files. This package\n  is tested to work on Shuttle Radar Topography Mission (SRTM) DTED files\n  (as far as I can tell these are the only publicly available DTED files).\n  This can be used as a library to parse these files into `numpy` arrays\n  and additionally exposes a CLI that can be used to investigate individual\n  DTED files.\n  \nFor more information and resources about the DTED file format see the\n  end of the README.  \n\n## How to install\nYou can install this as a normal python package using `pip`\n```bash\npip install dted\n``` \n\n## How to use\n\nThe following example code will parse DTED file checked into this repository\n  for testing.\n\n### As a library\n\nParsing a DTED file into a numpy array is as simple as:\n```python\nimport numpy as np\nfrom pathlib import Path\nfrom dted import Tile\n\ndted_file = Path("test/data/n41_w071_1arc_v3.dt2")\ntile = Tile(dted_file)\nassert isinstance(tile.data, np.ndarray)\n```\n\nAdditionally, you can access the metadata of the DTED file (the User Header \n  Label, Data Set Identification, and Accuracy Description records) easily.\n\n```python\nfrom pathlib import Path\nfrom dted import Tile\n\ndted_file = Path("test/data/n41_w071_1arc_v3.dt2")\ntile = Tile(dted_file)\nprint(tile.dsi.south_west_corner)\n```\n\nParsing entire DTED files has been heavily optimized, but does still take\n  a little bit of time. On my machine (2014 MacbookPro) parsing the 25MB\n  example file take about 120 ms. However, if you only need to look up\n  specific terrain elevations within a DTED file you do not need to parse\n  the entire file. Doing the following takes <1ms on my machine: \n\n```python\nfrom pathlib import Path\nfrom dted import LatLon, Tile\n\ndted_file = Path("test/data/n41_w071_1arc_v3.dt2")\ntile = Tile(dted_file, in_memory=False)\nprint(tile.get_elevation(LatLon(latitude=41.5, longitude=-70.5)))\n```\n\nIf for some reason you _really_ need to eke out every bit of performance, and\n  you thoroughly trust your DTED data, you speed up the data parsing by\n  skipping the checksum verification. Doing the following takes about 75 ms\n  on my machine:\n  \n```python\nimport numpy as np\nfrom pathlib import Path\nfrom dted import Tile\n\ndted_file = Path("test/data/n41_w071_1arc_v3.dt2")\ntile = Tile(dted_file, in_memory=False)\ntile.load_data(perform_checksum=False)\n\nassert isinstance(tile.data, np.ndarray)\n```\n\nThe final functionality the `dted.Tile` class offers is to easily check if\n  a coordinate location is contained within the DTED file. This also does\n  not require that the DTED data is fully loaded into memory:\n\n```python\nfrom pathlib import Path\nfrom dted import LatLon, Tile\n\ndted_file = Path("test/data/n41_w071_1arc_v3.dt2")\ntile = Tile(dted_file, in_memory=False)\n\nassert LatLon(latitude=41.5, longitude=-70.25) in tile\n```\n\n## As a CLI\n\nInstalling this package into an activated virtual environment also exposes\n  the `dted` terminal command. This provides three pieces of functionality:\n1. See report of the metadata of the DTED file.\n2. Lookup terrain elevation at a specific point within the DTED file.\n3. Display and ASCII representation of the DTED file in your terminal.\n\nTo get a report of the file metadata:\n```bash\n(.venv) user@machine$ dted test/data/n41_w071_1arc_v3.dt2 \nFile Path:          test/data/n41_w071_1arc_v3.dt2 (24 MB)\nProduct Level:      DTED2\nSecurity Code:      U\nCompilation Date:   02/2000\nMaintenance Date:   \nDatums (V/H):       E96/WGS84\n\n    (42.0N,71.0W)      (42.0N,70.0W)\n          NW --------------- NE     \n          |                   |     \n          |                   |     \n          |                   |     \n          |                   |     \n          |                   |     \n          |                   |     \n          SW --------------- SE     \n    (41.0N,71.0W)      (41.0N,70.0W)\n\nOrigin:                 (41.0N,71.0W)\nResolution (lat/lon):   1.0"/1.0"\nAccuracy (V/H):         6m/13m\n```\n\nTo lookup terrain elevation at a specific point:\n```bash\n(.venv) user@machine$ dted test/data/n41_w071_1arc_v3.dt2 --location 41.7 -70.4\n51.0 meters\n```\n\nTo display the DTED file in your terminal:\n```bash\n(.venv) user@machine$ dted test/data/n41_w071_1arc_v3.dt2 --display\n```\nThis will attempt to create an ASCII representation of the DTED file within\n  your terminal at the best resolution possible. Increasing the size of your\n  terminal window or zooming out your terminal window will increase the \n  resolution of the chart:\n  \n![Normal Resolution Image](.images/normal_resolution.png)\n\n![High Resolution Image](.images/high_resolution.png)\n\nWhy did I add this feature? Why not?\n\nIf you want to plot this data like a sane person, you can use the following\n example code with the `matplotlib` package (not included)\n\n```python\nimport matplotlib.pyplot as plt\nfrom pathlib import Path\nfrom dted import Tile\n\ndted_file = Path("test/data/n41_w071_1arc_v3.dt2")\ntile = Tile(dted_file)\nplt.imshow(tile.data.T[::-1], cmap="hot")\n```\n\n## The DTED file format\n\nThis parser was created using the specification provided here:\n```\nhttps://www.dlr.de/eoc/Portaldata/60/Resources/dokumente/7_sat_miss/SRTM-XSAR-DEM-DTED-1.1.pdf\n```\n\nSome things to be aware of with the DTED file format:\n1. Some DTED files contain "void" values for data points where elevation\n  data is not known (such as over bodies of water). An example of such a\n  file can be found at `test/data/n00_e006_3arc_v2.dt1`. This package\n  will emit a warning if void data is found, and the definition of the\n  void value can be found in `dted.definitions.VOID_DATA_VALUE`.\n2. The DTED data is structured along longitudinal lines. Therefore, when\n  accessing the data within the `numpy` array the rows correspond to \n  longitude and the columns correspond to latitude. This may seem backwards\n  to your intuition, i.e. you would access the elevation at a coordinate\n  point with `tile.data[longitude_index, latitude_index]`.\n3. Elevation within the DTED file is encoded using "signed magnitude"\n  notation. This has no effect on a user of this package interacting with\n  the parsed terrain elevation data, but it does slow down the parsing of \n  this data as I do not know of an optimized method of parsing signed\n  magnitude data in python. If someone knows how to do this, this parsing\n  library could become even faster. \n\n### Where to find DTED data\n\nPublicly available DTED data is relatively hard to find and access, \n  but it can be done. The DTED files I used for testing and developing \n  this package come from `https://earthexplorer.usgs.gov/`.\n\nThis EarthExplorer app provided by the USGS provides an interface to \n  download many types of terrain data, including the SRTM DTED data.\n  However, you need to make an account with them in order to perform the\n  download, and I\'m unsure of a way to use their machine-to-machine API\n  to automate downloading data.\n\n### Contributing\n\nContributions are absolutely encouraged! To develop on this project you\n  need to install the `poetry` package manager.\n \nClone the repo:\n```bash\nuser@machine$ git clone https://github.com/bbonenfant/dted\n```\n\nCreate and activate the virtual environment:\n```bash\nuser@machine$ poetry install && source .venv/bin/activate\n```\n\nTo run the tests:\n```bash\n(.venv) user@machine$ pytest .\n```\n\nIf you are getting `BLACK` errors from pytest, run the `black` code formatter:\n```bash\n(.venv) user@machine$ black .\n```',
    'author': 'Ben Bonenfant',
    'author_email': 'bonenfan5ben@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bbonenfant/dted',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
