import time
from string import Template
import datetime
import yaml
from dbstream import DBStream
from tabulate import tabulate

from pydatasource.core.documentation import document_treat_query
from pydatasource.core.snippet import treat_all_snippet
from dacktool import log_info, log_error

from pydatasource.core.environment_comparison import launch_test


def get_destination_tables_with_schema(queries_config, query_name, schema_name, environment):
    result = {"production": schema_name + "." + query_name}
    if environment != "production":
        result[environment] = schema_name + "_" + environment + "." + query_name
    if queries_config[query_name].get("query_params"):
        table_destination = queries_config[query_name].get("query_params").get("table_destination")
        if table_destination:
            if environment == "production":
                if isinstance(table_destination, dict):
                    result["production"] = table_destination["production"]
                else:
                    result["production"] = table_destination
            else:
                result["production"] = table_destination["production"]
                result[environment] = table_destination[environment]
    return result


class DataSource:
    def __init__(self, dbstream: DBStream, path_to_datasource_folder, schema_prefix=None, config_name='config',
                 layer_type='datasource'):
        """

        :param dbstream:
        :param path_to_datasource_folder: absolute path containing the last '/'
        """
        self.dbstream = dbstream
        self.path_to_datasource_folder = path_to_datasource_folder
        self.schema_prefix = schema_prefix
        self.config_name = config_name
        self.layer_type = layer_type

    def _build_folder_path(self, layer_name):
        return self.path_to_datasource_folder + 'layers/' + layer_name + "/"

    def _create_beautiful_view(self, table_name, schema_name):

        result = self.dbstream.get_data_type(table_name=table_name, schema_name=schema_name)

        columns_list = []
        for c in result:
            columns_list.append(""" "%s" as "%s" """ % (c["column_name"], c["column_name"].replace("_", " ")))

        view_name = '%s.%s_%s' % (schema_name, table_name, 'beautiful')
        columns = ','.join(columns_list)
        self.dbstream.create_view_from_columns(view_name, columns, schema_name, table_name)
        log_info(view_name + " view created")

    def _create_gds_view(self, table_name, schema_name):

        result = self.dbstream.get_data_type(table_name=table_name, schema_name=schema_name)

        columns_list = []
        for c in result:
            if c["data_type"] in ('timestamp without time zone',):
                columns_list.append("TO_CHAR(%s,'YYYYMMDD') as %s" % (c["column_name"], c["column_name"]))
            else:
                columns_list.append(c["column_name"])

        view_name = '%s.%s_%s' % (schema_name, table_name, 'gds')
        columns = ','.join(columns_list)
        self.dbstream.create_view_from_columns(view_name, columns, schema_name, table_name)
        log_info(view_name + " view created")

    def _get_query_list(self, layer_name, query_name=None):
        folder_path = self._build_folder_path(layer_name)
        config = yaml.load(open(folder_path + ("%s.yaml" % self.config_name)), Loader=yaml.FullLoader)
        queries = config.get("queries")
        schema_name = config.get("schema_name") if config.get("schema_name") else (self.layer_type + "_" + layer_name)
        schema_name = self.schema_prefix + '_' + schema_name if self.schema_prefix else schema_name
        if not queries:
            log_info("No queries set up in config file")
            return 0
        query_list = [query_name] if query_name else list(queries.keys())
        return query_list, queries, schema_name, folder_path

    def _filled_query(self, queries_config, query, folder_path, schema_name, layer_name, environment="production"):
        query_config = queries_config[query]
        if query_config.get("template"):
            query_template_file_name = query_config.get("template")
        else:
            query_template_file_name = query
        query_path = folder_path + "query/" + query_template_file_name + ".sql"
        query_params = query_config.get("query_params")
        query_template = Template(open(query_path).read())
        table_name = query
        dict_params = dict()
        dict_params["TABLE_NAME"] = "%s.%s" % (schema_name, table_name)
        dict_params["TABLE_NAME_TEMP"] = "%s_%s_temp" % (schema_name, table_name)
        if query_params:
            for params in query_params.keys():
                value = query_params[params]
                if isinstance(value, dict):
                    if environment != 'production':
                        value = value[environment]
                    else:
                        value = value["production"]
                if value == "now":
                    value = "'" + str(datetime.datetime.now())[:10] + "'"
                dict_params.update({params.upper(): value})
        dict_params.update(
            treat_all_snippet(
                datasource_path=self.path_to_datasource_folder,
                query_path=query_path,
                layer=layer_name,
                dict_params=dict_params)
        )
        filled_query = query_template.substitute(dict_params)
        return filled_query

    def compute(self, layer_name, query_name=None, environment="production"):
        query_list, queries_config, schema_name, folder_path = self._get_query_list(layer_name, query_name=query_name)

        for query in query_list:
            log_info("Layer: %s | Query started: %s | Environment: %s" % (layer_name, query, environment))
            filled_query = self._filled_query(
                queries_config=queries_config,
                query=query,
                folder_path=folder_path,
                schema_name=schema_name if environment == 'production' else (schema_name + "_" + environment),
                layer_name=layer_name,
                environment=environment
            )
            destination_tables_with_schema = get_destination_tables_with_schema(
                schema_name=schema_name,
                queries_config=queries_config,
                query_name=query,
                environment=environment
            )
            if environment != "production":
                log_info(filled_query)
            try:
                self.dbstream.execute_query(filled_query)
            except Exception as e:
                if "schema" in str(e):
                    self.dbstream.create_schema(destination_tables_with_schema[environment].split(".")[0])
                    self.dbstream.execute_query(filled_query)
                else:
                    sec_before_retry = 15
                    log_error(str(e))
                    log_info("Retry in %s s..." % str(sec_before_retry))
                    time.sleep(sec_before_retry)
                    self.dbstream.execute_query(filled_query)

            # LOG NORMAL
            if environment == 'production':
                log_info(destination_tables_with_schema["production"] + " created")

            if environment == 'production' and queries_config[query].get("beautiful_view"):
                self._create_beautiful_view(
                    schema_name=destination_tables_with_schema["production"].split(".")[0],
                    table_name=destination_tables_with_schema["production"].split(".")[1],
                )
            if environment == 'production' and queries_config[query].get("gds"):
                self._create_gds_view(
                    schema_name=destination_tables_with_schema["production"].split(".")[0],
                    table_name=destination_tables_with_schema["production"].split(".")[1],
                )

        if environment != 'production':
            self.compute_comparison(
                schema_name=schema_name,
                tables_list=query_list,
                queries_config=queries_config,
                environment=environment)

        return 0

    def compute_comparison(self, schema_name, tables_list, environment, queries_config):
        print("=============================")
        print("ENVIRONMENTS COMPARISON RESULTS")
        print("=============================")
        for table in tables_list:

            destination_tables_with_schema = get_destination_tables_with_schema(
                schema_name=schema_name,
                queries_config=queries_config,
                query_name=table,
                environment=environment
            )

            print("Result of %s" % destination_tables_with_schema[environment])

            test_where_clause = queries_config[table].get("test_where_clause")

            # Prod
            prod_data_types = self.dbstream.get_data_type(
                table_name=destination_tables_with_schema["production"].split(".")[1],
                schema_name=destination_tables_with_schema["production"].split(".")[0]
            )
            if prod_data_types is None:
                print("table %s is new" % destination_tables_with_schema["production"])
                print("=============================")
                continue
            prod_result_values = launch_test(
                dbstream=self.dbstream,
                schema_name=destination_tables_with_schema["production"].split(".")[0],
                table=destination_tables_with_schema["production"].split(".")[1],
                data_types=prod_data_types,
                test_where_clause=test_where_clause
            )

            # Other Environment
            env_data_types = self.dbstream.get_data_type(
                table_name=destination_tables_with_schema[environment].split(".")[1],
                schema_name=destination_tables_with_schema[environment].split(".")[0]
            )
            env_result_values = launch_test(
                dbstream=self.dbstream,
                schema_name=destination_tables_with_schema[environment].split(".")[0],
                table=destination_tables_with_schema[environment].split(".")[1],
                data_types=env_data_types,
                test_where_clause=test_where_clause
            )

            main_dict = {}
            for p in prod_result_values[0].keys():
                main_dict[p] = {"prod_value": prod_result_values[0][p]}

            for s in env_result_values[0].keys():
                if main_dict.get(s):
                    main_dict[s]["env_value"] = env_result_values[0][s]
                else:
                    main_dict[s] = {"env_value": env_result_values[0][s]}

            headers = ["Metric", "Prod", environment]
            values = []
            for key in main_dict:
                val = [key]
                if main_dict[key].get("prod_value"):
                    val.append(main_dict[key].get("prod_value"))
                else:
                    val.append("===NEW METRIC===")
                if main_dict[key].get("env_value") and main_dict[key].get("prod_value"):
                    diff = main_dict[key].get("env_value") - main_dict[key].get("prod_value")
                    if diff == 0:
                        val.append("OK SAME VALUE")
                    else:
                        val.append("DIFF (env-prod) : %s" % str(diff))
                elif main_dict[key].get("env_value"):
                    val.append(main_dict[key].get("env_value"))
                else:
                    val.append("===DISAPPEARED===")
                values.append(val)

            print(tabulate(values, headers=headers, tablefmt="fancy_grid", floatfmt=".2f"))
            print("=============================")

    def function_compute(self, layer_name, environment='production'):
        def f():
            self.compute(layer_name, environment)

        return f

    def print_filled_query(self, layer_name, query_name=None, environment="production"):
        query_list, queries, schema_name, folder_path = self._get_query_list(layer_name, query_name=query_name)
        for query in query_list:
            filled_query = self._filled_query(
                queries_config=queries,
                query=query,
                folder_path=folder_path,
                schema_name=schema_name,
                layer_name=layer_name,
                environment=environment
            )
            with open('sandbox_' + layer_name + '_' + query_name + '.sql', 'w') as f:
                f.write(filled_query)

    def doc(self, layer_name, query_name=None, environment="production"):
        r = []
        query_list, queries_config, schema_name, folder_path = self._get_query_list(layer_name, query_name=query_name)
        for query in query_list:
            filled_query = self._filled_query(
                queries_config=queries_config,
                query=query,
                folder_path=folder_path,
                schema_name=schema_name,
                layer_name=layer_name,
                environment=environment
            )
            r = r + document_treat_query(
                filled_query=filled_query,
                schema_name=schema_name,
                table_name=get_destination_tables_with_schema(
                    queries_config=queries_config,
                    query_name=query,
                    schema_name=schema_name,
                    environment=environment
                ))
        return r
