"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Default = void 0;
/**
 * Provides default values for certain regional information points.
 */
class Default {
    constructor() { }
    /**
     * Computes a "standard" AWS Service principal for a given service, region and suffix. This is useful for example when
     * you need to compute a service principal name, but you do not have a synthesize-time region literal available (so
     * all you have is `{ "Ref": "AWS::Region" }`). This way you get the same defaulting behavior that is normally used
     * for built-in data.
     *
     * @param service   the name of the service (s3, s3.amazonaws.com, ...)
     * @param region    the region in which the service principal is needed.
     * @param urlSuffix the URL suffix for the partition in which the region is located.
     */
    static servicePrincipal(service, region, urlSuffix) {
        const matches = service.match(/^([^.]+)(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))?$/);
        if (!matches) {
            // Return "service" if it does not look like any of the following:
            // - s3
            // - s3.amazonaws.com
            // - s3.amazonaws.com.cn
            // - s3.c2s.ic.gov
            // - s3.sc2s.sgov.gov
            return service;
        }
        service = matches[1]; // Simplify the service name down to something like "s3"
        // Exceptions for Service Principals in us-iso-*
        const US_ISO_EXCEPTIONS = new Set([
            'cloudhsm',
            'config',
            'states',
            'workspaces',
        ]);
        // Exceptions for Service Principals in us-isob-*
        const US_ISOB_EXCEPTIONS = new Set([
            'dms',
            'states',
        ]);
        // Account for idiosyncratic Service Principals in `us-iso-*` regions
        if (region.startsWith('us-iso-') && US_ISO_EXCEPTIONS.has(service)) {
            switch (service) {
                // Services with universal principal
                case ('states'):
                    return `${service}.amazonaws.com`;
                // Services with a partitional principal
                default:
                    return `${service}.${urlSuffix}`;
            }
        }
        // Account for idiosyncratic Service Principals in `us-isob-*` regions
        if (region.startsWith('us-isob-') && US_ISOB_EXCEPTIONS.has(service)) {
            switch (service) {
                // Services with universal principal
                case ('states'):
                    return `${service}.amazonaws.com`;
                // Services with a partitional principal
                default:
                    return `${service}.${urlSuffix}`;
            }
        }
        switch (service) {
            // Services with a regional AND partitional principal
            case 'codedeploy':
            case 'logs':
                return `${service}.${region}.${urlSuffix}`;
            // Services with a regional principal
            case 'states':
                return `${service}.${region}.amazonaws.com`;
            // Services with a partitional principal
            case 'ec2':
                return `${service}.${urlSuffix}`;
            // Services with a universal principal across all regions/partitions (the default case)
            default:
                return `${service}.amazonaws.com`;
        }
    }
}
exports.Default = Default;
/**
 * The default value for a VPC Endpoint Service name prefix, useful if you do
 * not have a synthesize-time region literal available (all you have is
 * `{ "Ref": "AWS::Region" }`)
 */
Default.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'com.amazonaws.vpce';
//# sourceMappingURL=data:application/json;base64,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