"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackOutput = exports.CdkStage = void 0;
const cpactions = require("../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const asset_manifest_1 = require("./private/asset-manifest");
const toposort_1 = require("./private/toposort");
/**
 * Stage in a CdkPipeline
 *
 * You don't need to instantiate this class directly. Use
 * `cdkPipeline.addStage()` instead.
 */
class CdkStage extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._nextSequentialRunOrder = 1; // Must start at 1 eh
        this._manualApprovalCounter = 1;
        this.stacksToDeploy = new Array();
        this._prepared = false;
        this.stageName = props.stageName;
        this.pipelineStage = props.pipelineStage;
        this.cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        this.host = props.host;
        core_1.Aspects.of(this).add({ visit: () => this.prepareStage() });
    }
    /**
     * Add all stacks in the application Stage to this stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplication(appStage, options = {}) {
        const asm = appStage.synth();
        const sortedTranches = toposort_1.topologicalSort(asm.stacks, stack => stack.id, stack => stack.dependencies.map(d => d.id));
        for (const stacks of sortedTranches) {
            const runOrder = this.nextSequentialRunOrder(2); // We need 2 actions
            let executeRunOrder = runOrder + 1;
            // If we need to insert a manual approval action, then what's the executeRunOrder
            // now is where we add a manual approval step, and we allocate 1 more runOrder
            // for the execute.
            if (options.manualApprovals) {
                this.addManualApprovalAction({ runOrder: executeRunOrder });
                executeRunOrder = this.nextSequentialRunOrder();
            }
            // These don't have a dependency on each other, so can all be added in parallel
            for (const stack of stacks) {
                this.addStackArtifactDeployment(stack, { runOrder, executeRunOrder });
            }
        }
    }
    /**
     * Add a deployment action based on a stack artifact
     */
    addStackArtifactDeployment(stackArtifact, options = {}) {
        var _a, _b;
        // Get all assets manifests and add the assets in 'em to the asset publishing stage.
        this.publishAssetDependencies(stackArtifact);
        // Remember for later, see 'prepare()'
        // We know that deploying a stack is going to take up 2 runorder slots later on.
        const runOrder = (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(2);
        const executeRunOrder = (_b = options.executeRunOrder) !== null && _b !== void 0 ? _b : runOrder + 1;
        this.stacksToDeploy.push({
            prepareRunOrder: runOrder,
            executeRunOrder,
            stackArtifact,
        });
        this.advanceRunOrderPast(runOrder);
        this.advanceRunOrderPast(executeRunOrder);
    }
    /**
     * Add a manual approval action
     *
     * If you need more flexibility than what this method offers,
     * use `addAction` with a `ManualApprovalAction`.
     */
    addManualApprovalAction(options = {}) {
        var _a;
        let actionName = options.actionName;
        if (!actionName) {
            actionName = `ManualApproval${this._manualApprovalCounter > 1 ? this._manualApprovalCounter : ''}`;
            this._manualApprovalCounter += 1;
        }
        this.addActions(new cpactions.ManualApprovalAction({
            actionName,
            runOrder: (_a = options.runOrder) !== null && _a !== void 0 ? _a : this.nextSequentialRunOrder(),
        }));
    }
    /**
     * Add one or more CodePipeline Actions
     *
     * You need to make sure it is created with the right runOrder. Call `nextSequentialRunOrder()`
     * for every action to get actions to execute in sequence.
     */
    addActions(...actions) {
        for (const action of actions) {
            this.pipelineStage.addAction(action);
        }
    }
    /**
     * Return the runOrder number necessary to run the next Action in sequence with the rest
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    nextSequentialRunOrder(count = 1) {
        const ret = this._nextSequentialRunOrder;
        this._nextSequentialRunOrder += count;
        return ret;
    }
    /**
     * Whether this Stage contains an action to deploy the given stack, identified by its artifact ID
     */
    deploysStack(artifactId) {
        return this.stacksToDeploy.map(s => s.stackArtifact.id).includes(artifactId);
    }
    /**
     * Actually add all the DeployStack actions to the stage.
     *
     * We do this late because before we can render the actual DeployActions,
     * we need to know whether or not we need to capture the stack outputs.
     *
     * FIXME: This is here because Actions are immutable and can't be reordered
     * after creation, nor is there a way to specify relative priorities, which
     * is a limitation that we should take away in the base library.
     */
    prepareStage() {
        // FIXME: Make sure this only gets run once. There seems to be an issue in the reconciliation
        // loop that may trigger this more than once if it throws an error somewhere, and the exception
        // that gets thrown here will then override the actual failure.
        if (this._prepared) {
            return;
        }
        this._prepared = true;
        for (const { prepareRunOrder, stackArtifact, executeRunOrder } of this.stacksToDeploy) {
            const artifact = this.host.stackOutputArtifact(stackArtifact.id);
            this.pipelineStage.addAction(actions_1.DeployCdkStackAction.fromStackArtifact(this, stackArtifact, {
                baseActionName: this.simplifyStackName(stackArtifact.stackName),
                cloudAssemblyInput: this.cloudAssemblyArtifact,
                output: artifact,
                outputFileName: artifact ? 'outputs.json' : undefined,
                prepareRunOrder,
                executeRunOrder,
            }));
        }
    }
    /**
     * Advance the runorder counter so that the next sequential number is higher than the given one
     */
    advanceRunOrderPast(lastUsed) {
        this._nextSequentialRunOrder = Math.max(lastUsed + 1, this._nextSequentialRunOrder);
    }
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    simplifyStackName(s) {
        return stripPrefix(s, `${this.stageName}-`);
    }
    /**
     * Make sure all assets depended on by this stack are published in this pipeline
     *
     * Taking care to exclude the stack template itself -- it is being published
     * as an asset because the CLI needs to know the asset publishing role when
     * pushing the template to S3, but in the case of CodePipeline we always
     * reference the template from the artifact bucket.
     *
     * (NOTE: this is only true for top-level stacks, not nested stacks. Nested
     * Stack templates are always published as assets).
     */
    publishAssetDependencies(stackArtifact) {
        const assetManifests = stackArtifact.dependencies.filter(isAssetManifest);
        for (const manifestArtifact of assetManifests) {
            const manifest = asset_manifest_1.AssetManifestReader.fromFile(manifestArtifact.file);
            for (const entry of manifest.entries) {
                let assetType;
                if (entry instanceof asset_manifest_1.DockerImageManifestEntry) {
                    assetType = actions_1.AssetType.DOCKER_IMAGE;
                }
                else if (entry instanceof asset_manifest_1.FileManifestEntry) {
                    // Don't publishg the template for this stack
                    if (entry.source.packaging === 'file' && entry.source.path === stackArtifact.templateFile) {
                        continue;
                    }
                    assetType = actions_1.AssetType.FILE;
                }
                else {
                    throw new Error(`Unrecognized asset type: ${entry.type}`);
                }
                this.host.publishAsset({
                    assetManifestPath: manifestArtifact.file,
                    assetId: entry.id.assetId,
                    assetSelector: entry.id.toString(),
                    assetType,
                });
            }
        }
    }
}
exports.CdkStage = CdkStage;
/**
 * A single output of a Stack
 */
class StackOutput {
    /**
     * Build a StackOutput from a known artifact and an output name
     */
    constructor(artifactFile, outputName) {
        this.artifactFile = artifactFile;
        this.outputName = outputName;
    }
}
exports.StackOutput = StackOutput;
function stripPrefix(s, prefix) {
    return s.startsWith(prefix) ? s.substr(prefix.length) : s;
}
function isAssetManifest(s) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return s instanceof cxapi.AssetManifestArtifact;
    return s.constructor.name === 'AssetManifestArtifact';
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhZ2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGFnZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSw0REFBNEQsQ0FBQyxvRUFBb0U7QUFDakkscUNBQXVELENBQUMsZ0RBQWdEO0FBRXhHLHVDQUE0RDtBQUM1RCw2REFBNEc7QUFDNUcsaURBQXFEO0FBc0JyRDs7Ozs7R0FLRztBQUNILE1BQWEsUUFBUyxTQUFRLGdCQUFTO0lBU25DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBb0I7UUFDMUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQVRiLDRCQUF1QixHQUFHLENBQUMsQ0FBQyxDQUFDLHFCQUFxQjtRQUNsRCwyQkFBc0IsR0FBRyxDQUFDLENBQUM7UUFHbEIsbUJBQWMsR0FBRyxJQUFJLEtBQUssRUFBc0IsQ0FBQztRQUcxRCxjQUFTLEdBQUcsS0FBSyxDQUFDO1FBR3RCLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUNqQyxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7UUFDekMsSUFBSSxDQUFDLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQztRQUN6RCxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDdkIsY0FBTyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7T0FVRztJQUNJLGNBQWMsQ0FBQyxRQUFlLEVBQUUsVUFBMkIsRUFBRTtRQUNoRSxNQUFNLEdBQUcsR0FBRyxRQUFRLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDN0IsTUFBTSxjQUFjLEdBQUcsMEJBQWUsQ0FBQyxHQUFHLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDbEgsS0FBSyxNQUFNLE1BQU0sSUFBSSxjQUFjLEVBQUU7WUFDakMsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsb0JBQW9CO1lBQ3JFLElBQUksZUFBZSxHQUFHLFFBQVEsR0FBRyxDQUFDLENBQUM7WUFDbkMsaUZBQWlGO1lBQ2pGLDhFQUE4RTtZQUM5RSxtQkFBbUI7WUFDbkIsSUFBSSxPQUFPLENBQUMsZUFBZSxFQUFFO2dCQUN6QixJQUFJLENBQUMsdUJBQXVCLENBQUMsRUFBRSxRQUFRLEVBQUUsZUFBZSxFQUFFLENBQUMsQ0FBQztnQkFDNUQsZUFBZSxHQUFHLElBQUksQ0FBQyxzQkFBc0IsRUFBRSxDQUFDO2FBQ25EO1lBQ0QsK0VBQStFO1lBQy9FLEtBQUssTUFBTSxLQUFLLElBQUksTUFBTSxFQUFFO2dCQUN4QixJQUFJLENBQUMsMEJBQTBCLENBQUMsS0FBSyxFQUFFLEVBQUUsUUFBUSxFQUFFLGVBQWUsRUFBRSxDQUFDLENBQUM7YUFDekU7U0FDSjtJQUNMLENBQUM7SUFDRDs7T0FFRztJQUNJLDBCQUEwQixDQUFDLGFBQWdELEVBQUUsVUFBMkIsRUFBRTs7UUFDN0csb0ZBQW9GO1FBQ3BGLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUM3QyxzQ0FBc0M7UUFDdEMsZ0ZBQWdGO1FBQ2hGLE1BQU0sUUFBUSxTQUFHLE9BQU8sQ0FBQyxRQUFRLG1DQUFJLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNwRSxNQUFNLGVBQWUsU0FBRyxPQUFPLENBQUMsZUFBZSxtQ0FBSSxRQUFRLEdBQUcsQ0FBQyxDQUFDO1FBQ2hFLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDO1lBQ3JCLGVBQWUsRUFBRSxRQUFRO1lBQ3pCLGVBQWU7WUFDZixhQUFhO1NBQ2hCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNuQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUNEOzs7OztPQUtHO0lBQ0ksdUJBQXVCLENBQUMsVUFBb0MsRUFBRTs7UUFDakUsSUFBSSxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztRQUNwQyxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ2IsVUFBVSxHQUFHLGlCQUFpQixJQUFJLENBQUMsc0JBQXNCLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQ25HLElBQUksQ0FBQyxzQkFBc0IsSUFBSSxDQUFDLENBQUM7U0FDcEM7UUFDRCxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksU0FBUyxDQUFDLG9CQUFvQixDQUFDO1lBQy9DLFVBQVU7WUFDVixRQUFRLFFBQUUsT0FBTyxDQUFDLFFBQVEsbUNBQUksSUFBSSxDQUFDLHNCQUFzQixFQUFFO1NBQzlELENBQUMsQ0FBQyxDQUFDO0lBQ1IsQ0FBQztJQUNEOzs7OztPQUtHO0lBQ0ksVUFBVSxDQUFDLEdBQUcsT0FBK0I7UUFDaEQsS0FBSyxNQUFNLE1BQU0sSUFBSSxPQUFPLEVBQUU7WUFDMUIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDeEM7SUFDTCxDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCLENBQUMsUUFBZ0IsQ0FBQztRQUMzQyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUM7UUFDekMsSUFBSSxDQUFDLHVCQUF1QixJQUFJLEtBQUssQ0FBQztRQUN0QyxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7SUFDRDs7T0FFRztJQUNJLFlBQVksQ0FBQyxVQUFrQjtRQUNsQyxPQUFPLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUNEOzs7Ozs7Ozs7T0FTRztJQUNLLFlBQVk7UUFDaEIsNkZBQTZGO1FBQzdGLCtGQUErRjtRQUMvRiwrREFBK0Q7UUFDL0QsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ2hCLE9BQU87U0FDVjtRQUNELElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDO1FBQ3RCLEtBQUssTUFBTSxFQUFFLGVBQWUsRUFBRSxhQUFhLEVBQUUsZUFBZSxFQUFFLElBQUksSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUNuRixNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNqRSxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyw4QkFBb0IsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO2dCQUNyRixjQUFjLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLGFBQWEsQ0FBQyxTQUFTLENBQUM7Z0JBQy9ELGtCQUFrQixFQUFFLElBQUksQ0FBQyxxQkFBcUI7Z0JBQzlDLE1BQU0sRUFBRSxRQUFRO2dCQUNoQixjQUFjLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQ3JELGVBQWU7Z0JBQ2YsZUFBZTthQUNsQixDQUFDLENBQUMsQ0FBQztTQUNQO0lBQ0wsQ0FBQztJQUNEOztPQUVHO0lBQ0ssbUJBQW1CLENBQUMsUUFBZ0I7UUFDeEMsSUFBSSxDQUFDLHVCQUF1QixHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsUUFBUSxHQUFHLENBQUMsRUFBRSxJQUFJLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUN4RixDQUFDO0lBQ0Q7O09BRUc7SUFDSyxpQkFBaUIsQ0FBQyxDQUFTO1FBQy9CLE9BQU8sV0FBVyxDQUFDLENBQUMsRUFBRSxHQUFHLElBQUksQ0FBQyxTQUFTLEdBQUcsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFDRDs7Ozs7Ozs7OztPQVVHO0lBQ0ssd0JBQXdCLENBQUMsYUFBZ0Q7UUFDN0UsTUFBTSxjQUFjLEdBQUcsYUFBYSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDLENBQUM7UUFDMUUsS0FBSyxNQUFNLGdCQUFnQixJQUFJLGNBQWMsRUFBRTtZQUMzQyxNQUFNLFFBQVEsR0FBRyxvQ0FBbUIsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDckUsS0FBSyxNQUFNLEtBQUssSUFBSSxRQUFRLENBQUMsT0FBTyxFQUFFO2dCQUNsQyxJQUFJLFNBQW9CLENBQUM7Z0JBQ3pCLElBQUksS0FBSyxZQUFZLHlDQUF3QixFQUFFO29CQUMzQyxTQUFTLEdBQUcsbUJBQVMsQ0FBQyxZQUFZLENBQUM7aUJBQ3RDO3FCQUNJLElBQUksS0FBSyxZQUFZLGtDQUFpQixFQUFFO29CQUN6Qyw2Q0FBNkM7b0JBQzdDLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxLQUFLLGFBQWEsQ0FBQyxZQUFZLEVBQUU7d0JBQ3ZGLFNBQVM7cUJBQ1o7b0JBQ0QsU0FBUyxHQUFHLG1CQUFTLENBQUMsSUFBSSxDQUFDO2lCQUM5QjtxQkFDSTtvQkFDRCxNQUFNLElBQUksS0FBSyxDQUFDLDRCQUE0QixLQUFLLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztpQkFDN0Q7Z0JBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUM7b0JBQ25CLGlCQUFpQixFQUFFLGdCQUFnQixDQUFDLElBQUk7b0JBQ3hDLE9BQU8sRUFBRSxLQUFLLENBQUMsRUFBRSxDQUFDLE9BQU87b0JBQ3pCLGFBQWEsRUFBRSxLQUFLLENBQUMsRUFBRSxDQUFDLFFBQVEsRUFBRTtvQkFDbEMsU0FBUztpQkFDWixDQUFDLENBQUM7YUFDTjtTQUNKO0lBQ0wsQ0FBQztDQUNKO0FBaE1ELDRCQWdNQztBQWtCRDs7R0FFRztBQUNILE1BQWEsV0FBVztJQVNwQjs7T0FFRztJQUNILFlBQVksWUFBdUMsRUFBRSxVQUFrQjtRQUNuRSxJQUFJLENBQUMsWUFBWSxHQUFHLFlBQVksQ0FBQztRQUNqQyxJQUFJLENBQUMsVUFBVSxHQUFHLFVBQVUsQ0FBQztJQUNqQyxDQUFDO0NBQ0o7QUFoQkQsa0NBZ0JDO0FBQ0QsU0FBUyxXQUFXLENBQUMsQ0FBUyxFQUFFLE1BQWM7SUFDMUMsT0FBTyxDQUFDLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQzlELENBQUM7QUFDRCxTQUFTLGVBQWUsQ0FBQyxDQUFzQjtJQUMzQywwRUFBMEU7SUFDMUUsbURBQW1EO0lBQ25ELE9BQU8sQ0FBQyxDQUFDLFdBQVcsQ0FBQyxJQUFJLEtBQUssdUJBQXVCLENBQUM7QUFDMUQsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvZGVwaXBlbGluZSBmcm9tIFwiLi4vLi4vYXdzLWNvZGVwaXBlbGluZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZSdcbmltcG9ydCAqIGFzIGNwYWN0aW9ucyBmcm9tIFwiLi4vLi4vYXdzLWNvZGVwaXBlbGluZS1hY3Rpb25zXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtY29kZXBpcGVsaW5lLWFjdGlvbnMnXG5pbXBvcnQgeyBDb25zdHJ1Y3QsIFN0YWdlLCBBc3BlY3RzIH0gZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgKiBhcyBjeGFwaSBmcm9tIFwiLi4vLi4vY3gtYXBpXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jeC1hcGknXG5pbXBvcnQgeyBBc3NldFR5cGUsIERlcGxveUNka1N0YWNrQWN0aW9uIH0gZnJvbSAnLi9hY3Rpb25zJztcbmltcG9ydCB7IEFzc2V0TWFuaWZlc3RSZWFkZXIsIERvY2tlckltYWdlTWFuaWZlc3RFbnRyeSwgRmlsZU1hbmlmZXN0RW50cnkgfSBmcm9tICcuL3ByaXZhdGUvYXNzZXQtbWFuaWZlc3QnO1xuaW1wb3J0IHsgdG9wb2xvZ2ljYWxTb3J0IH0gZnJvbSAnLi9wcml2YXRlL3RvcG9zb3J0Jztcbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgZm9yIGEgQ2RrU3RhZ2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDZGtTdGFnZVByb3BzIHtcbiAgICAvKipcbiAgICAgKiBOYW1lIG9mIHRoZSBzdGFnZSB0aGF0IHNob3VsZCBiZSBjcmVhdGVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3RhZ2VOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHVuZGVybHlpbmcgUGlwZWxpbmUgU3RhZ2UgYXNzb2NpYXRlZCB3aXRoIHRoaXNDZGtTdGFnZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBpcGVsaW5lU3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2U7XG4gICAgLyoqXG4gICAgICogVGhlIENvZGVQaXBlbGluZSBBcnRpZmFjdCB3aXRoIHRoZSBDbG91ZCBBc3NlbWJseVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNsb3VkQXNzZW1ibHlBcnRpZmFjdDogY29kZXBpcGVsaW5lLkFydGlmYWN0O1xuICAgIC8qKlxuICAgICAqIEZlYXR1cmVzIHRoZSBTdGFnZSBuZWVkcyBmcm9tIGl0cyBlbnZpcm9ubWVudFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGhvc3Q6IElTdGFnZUhvc3Q7XG59XG4vKipcbiAqIFN0YWdlIGluIGEgQ2RrUGlwZWxpbmVcbiAqXG4gKiBZb3UgZG9uJ3QgbmVlZCB0byBpbnN0YW50aWF0ZSB0aGlzIGNsYXNzIGRpcmVjdGx5LiBVc2VcbiAqIGBjZGtQaXBlbGluZS5hZGRTdGFnZSgpYCBpbnN0ZWFkLlxuICovXG5leHBvcnQgY2xhc3MgQ2RrU3RhZ2UgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAgIHByaXZhdGUgX25leHRTZXF1ZW50aWFsUnVuT3JkZXIgPSAxOyAvLyBNdXN0IHN0YXJ0IGF0IDEgZWhcbiAgICBwcml2YXRlIF9tYW51YWxBcHByb3ZhbENvdW50ZXIgPSAxO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcGlwZWxpbmVTdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGNsb3VkQXNzZW1ibHlBcnRpZmFjdDogY29kZXBpcGVsaW5lLkFydGlmYWN0O1xuICAgIHByaXZhdGUgcmVhZG9ubHkgc3RhY2tzVG9EZXBsb3kgPSBuZXcgQXJyYXk8RGVwbG95U3RhY2tDb21tYW5kPigpO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgc3RhZ2VOYW1lOiBzdHJpbmc7XG4gICAgcHJpdmF0ZSByZWFkb25seSBob3N0OiBJU3RhZ2VIb3N0O1xuICAgIHByaXZhdGUgX3ByZXBhcmVkID0gZmFsc2U7XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENka1N0YWdlUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgdGhpcy5zdGFnZU5hbWUgPSBwcm9wcy5zdGFnZU5hbWU7XG4gICAgICAgIHRoaXMucGlwZWxpbmVTdGFnZSA9IHByb3BzLnBpcGVsaW5lU3RhZ2U7XG4gICAgICAgIHRoaXMuY2xvdWRBc3NlbWJseUFydGlmYWN0ID0gcHJvcHMuY2xvdWRBc3NlbWJseUFydGlmYWN0O1xuICAgICAgICB0aGlzLmhvc3QgPSBwcm9wcy5ob3N0O1xuICAgICAgICBBc3BlY3RzLm9mKHRoaXMpLmFkZCh7IHZpc2l0OiAoKSA9PiB0aGlzLnByZXBhcmVTdGFnZSgpIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYWxsIHN0YWNrcyBpbiB0aGUgYXBwbGljYXRpb24gU3RhZ2UgdG8gdGhpcyBzdGFnZVxuICAgICAqXG4gICAgICogVGhlIGFwcGxpY2F0aW9uIGNvbnN0cnVjdCBzaG91bGQgc3ViY2xhc3MgYFN0YWdlYCBhbmQgY2FuIGNvbnRhaW4gYW55XG4gICAgICogbnVtYmVyIG9mIGBTdGFja3NgIGluc2lkZSBpdCB0aGF0IG1heSBoYXZlIGRlcGVuZGVuY3kgcmVsYXRpb25zaGlwc1xuICAgICAqIG9uIG9uZSBhbm90aGVyLlxuICAgICAqXG4gICAgICogQWxsIHN0YWNrcyBpbiB0aGUgYXBwbGljYXRpb24gd2lsbCBiZSBkZXBsb3llZCBpbiB0aGUgYXBwcm9wcmlhdGUgb3JkZXIsXG4gICAgICogYW5kIGFsbCBhc3NldHMgZm91bmQgaW4gdGhlIGFwcGxpY2F0aW9uIHdpbGwgYmUgYWRkZWQgdG8gdGhlIGFzc2V0XG4gICAgICogcHVibGlzaGluZyBzdGFnZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQXBwbGljYXRpb24oYXBwU3RhZ2U6IFN0YWdlLCBvcHRpb25zOiBBZGRTdGFnZU9wdGlvbnMgPSB7fSkge1xuICAgICAgICBjb25zdCBhc20gPSBhcHBTdGFnZS5zeW50aCgpO1xuICAgICAgICBjb25zdCBzb3J0ZWRUcmFuY2hlcyA9IHRvcG9sb2dpY2FsU29ydChhc20uc3RhY2tzLCBzdGFjayA9PiBzdGFjay5pZCwgc3RhY2sgPT4gc3RhY2suZGVwZW5kZW5jaWVzLm1hcChkID0+IGQuaWQpKTtcbiAgICAgICAgZm9yIChjb25zdCBzdGFja3Mgb2Ygc29ydGVkVHJhbmNoZXMpIHtcbiAgICAgICAgICAgIGNvbnN0IHJ1bk9yZGVyID0gdGhpcy5uZXh0U2VxdWVudGlhbFJ1bk9yZGVyKDIpOyAvLyBXZSBuZWVkIDIgYWN0aW9uc1xuICAgICAgICAgICAgbGV0IGV4ZWN1dGVSdW5PcmRlciA9IHJ1bk9yZGVyICsgMTtcbiAgICAgICAgICAgIC8vIElmIHdlIG5lZWQgdG8gaW5zZXJ0IGEgbWFudWFsIGFwcHJvdmFsIGFjdGlvbiwgdGhlbiB3aGF0J3MgdGhlIGV4ZWN1dGVSdW5PcmRlclxuICAgICAgICAgICAgLy8gbm93IGlzIHdoZXJlIHdlIGFkZCBhIG1hbnVhbCBhcHByb3ZhbCBzdGVwLCBhbmQgd2UgYWxsb2NhdGUgMSBtb3JlIHJ1bk9yZGVyXG4gICAgICAgICAgICAvLyBmb3IgdGhlIGV4ZWN1dGUuXG4gICAgICAgICAgICBpZiAob3B0aW9ucy5tYW51YWxBcHByb3ZhbHMpIHtcbiAgICAgICAgICAgICAgICB0aGlzLmFkZE1hbnVhbEFwcHJvdmFsQWN0aW9uKHsgcnVuT3JkZXI6IGV4ZWN1dGVSdW5PcmRlciB9KTtcbiAgICAgICAgICAgICAgICBleGVjdXRlUnVuT3JkZXIgPSB0aGlzLm5leHRTZXF1ZW50aWFsUnVuT3JkZXIoKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIC8vIFRoZXNlIGRvbid0IGhhdmUgYSBkZXBlbmRlbmN5IG9uIGVhY2ggb3RoZXIsIHNvIGNhbiBhbGwgYmUgYWRkZWQgaW4gcGFyYWxsZWxcbiAgICAgICAgICAgIGZvciAoY29uc3Qgc3RhY2sgb2Ygc3RhY2tzKSB7XG4gICAgICAgICAgICAgICAgdGhpcy5hZGRTdGFja0FydGlmYWN0RGVwbG95bWVudChzdGFjaywgeyBydW5PcmRlciwgZXhlY3V0ZVJ1bk9yZGVyIH0pO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIGRlcGxveW1lbnQgYWN0aW9uIGJhc2VkIG9uIGEgc3RhY2sgYXJ0aWZhY3RcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkU3RhY2tBcnRpZmFjdERlcGxveW1lbnQoc3RhY2tBcnRpZmFjdDogY3hhcGkuQ2xvdWRGb3JtYXRpb25TdGFja0FydGlmYWN0LCBvcHRpb25zOiBBZGRTdGFja09wdGlvbnMgPSB7fSkge1xuICAgICAgICAvLyBHZXQgYWxsIGFzc2V0cyBtYW5pZmVzdHMgYW5kIGFkZCB0aGUgYXNzZXRzIGluICdlbSB0byB0aGUgYXNzZXQgcHVibGlzaGluZyBzdGFnZS5cbiAgICAgICAgdGhpcy5wdWJsaXNoQXNzZXREZXBlbmRlbmNpZXMoc3RhY2tBcnRpZmFjdCk7XG4gICAgICAgIC8vIFJlbWVtYmVyIGZvciBsYXRlciwgc2VlICdwcmVwYXJlKCknXG4gICAgICAgIC8vIFdlIGtub3cgdGhhdCBkZXBsb3lpbmcgYSBzdGFjayBpcyBnb2luZyB0byB0YWtlIHVwIDIgcnVub3JkZXIgc2xvdHMgbGF0ZXIgb24uXG4gICAgICAgIGNvbnN0IHJ1bk9yZGVyID0gb3B0aW9ucy5ydW5PcmRlciA/PyB0aGlzLm5leHRTZXF1ZW50aWFsUnVuT3JkZXIoMik7XG4gICAgICAgIGNvbnN0IGV4ZWN1dGVSdW5PcmRlciA9IG9wdGlvbnMuZXhlY3V0ZVJ1bk9yZGVyID8/IHJ1bk9yZGVyICsgMTtcbiAgICAgICAgdGhpcy5zdGFja3NUb0RlcGxveS5wdXNoKHtcbiAgICAgICAgICAgIHByZXBhcmVSdW5PcmRlcjogcnVuT3JkZXIsXG4gICAgICAgICAgICBleGVjdXRlUnVuT3JkZXIsXG4gICAgICAgICAgICBzdGFja0FydGlmYWN0LFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5hZHZhbmNlUnVuT3JkZXJQYXN0KHJ1bk9yZGVyKTtcbiAgICAgICAgdGhpcy5hZHZhbmNlUnVuT3JkZXJQYXN0KGV4ZWN1dGVSdW5PcmRlcik7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIG1hbnVhbCBhcHByb3ZhbCBhY3Rpb25cbiAgICAgKlxuICAgICAqIElmIHlvdSBuZWVkIG1vcmUgZmxleGliaWxpdHkgdGhhbiB3aGF0IHRoaXMgbWV0aG9kIG9mZmVycyxcbiAgICAgKiB1c2UgYGFkZEFjdGlvbmAgd2l0aCBhIGBNYW51YWxBcHByb3ZhbEFjdGlvbmAuXG4gICAgICovXG4gICAgcHVibGljIGFkZE1hbnVhbEFwcHJvdmFsQWN0aW9uKG9wdGlvbnM6IEFkZE1hbnVhbEFwcHJvdmFsT3B0aW9ucyA9IHt9KSB7XG4gICAgICAgIGxldCBhY3Rpb25OYW1lID0gb3B0aW9ucy5hY3Rpb25OYW1lO1xuICAgICAgICBpZiAoIWFjdGlvbk5hbWUpIHtcbiAgICAgICAgICAgIGFjdGlvbk5hbWUgPSBgTWFudWFsQXBwcm92YWwke3RoaXMuX21hbnVhbEFwcHJvdmFsQ291bnRlciA+IDEgPyB0aGlzLl9tYW51YWxBcHByb3ZhbENvdW50ZXIgOiAnJ31gO1xuICAgICAgICAgICAgdGhpcy5fbWFudWFsQXBwcm92YWxDb3VudGVyICs9IDE7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5hZGRBY3Rpb25zKG5ldyBjcGFjdGlvbnMuTWFudWFsQXBwcm92YWxBY3Rpb24oe1xuICAgICAgICAgICAgYWN0aW9uTmFtZSxcbiAgICAgICAgICAgIHJ1bk9yZGVyOiBvcHRpb25zLnJ1bk9yZGVyID8/IHRoaXMubmV4dFNlcXVlbnRpYWxSdW5PcmRlcigpLFxuICAgICAgICB9KSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBvbmUgb3IgbW9yZSBDb2RlUGlwZWxpbmUgQWN0aW9uc1xuICAgICAqXG4gICAgICogWW91IG5lZWQgdG8gbWFrZSBzdXJlIGl0IGlzIGNyZWF0ZWQgd2l0aCB0aGUgcmlnaHQgcnVuT3JkZXIuIENhbGwgYG5leHRTZXF1ZW50aWFsUnVuT3JkZXIoKWBcbiAgICAgKiBmb3IgZXZlcnkgYWN0aW9uIHRvIGdldCBhY3Rpb25zIHRvIGV4ZWN1dGUgaW4gc2VxdWVuY2UuXG4gICAgICovXG4gICAgcHVibGljIGFkZEFjdGlvbnMoLi4uYWN0aW9uczogY29kZXBpcGVsaW5lLklBY3Rpb25bXSkge1xuICAgICAgICBmb3IgKGNvbnN0IGFjdGlvbiBvZiBhY3Rpb25zKSB7XG4gICAgICAgICAgICB0aGlzLnBpcGVsaW5lU3RhZ2UuYWRkQWN0aW9uKGFjdGlvbik7XG4gICAgICAgIH1cbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJuIHRoZSBydW5PcmRlciBudW1iZXIgbmVjZXNzYXJ5IHRvIHJ1biB0aGUgbmV4dCBBY3Rpb24gaW4gc2VxdWVuY2Ugd2l0aCB0aGUgcmVzdFxuICAgICAqXG4gICAgICogRklYTUU6IFRoaXMgaXMgaGVyZSBiZWNhdXNlIEFjdGlvbnMgYXJlIGltbXV0YWJsZSBhbmQgY2FuJ3QgYmUgcmVvcmRlcmVkXG4gICAgICogYWZ0ZXIgY3JlYXRpb24sIG5vciBpcyB0aGVyZSBhIHdheSB0byBzcGVjaWZ5IHJlbGF0aXZlIHByaW9yaXRpZXMsIHdoaWNoXG4gICAgICogaXMgYSBsaW1pdGF0aW9uIHRoYXQgd2Ugc2hvdWxkIHRha2UgYXdheSBpbiB0aGUgYmFzZSBsaWJyYXJ5LlxuICAgICAqL1xuICAgIHB1YmxpYyBuZXh0U2VxdWVudGlhbFJ1bk9yZGVyKGNvdW50OiBudW1iZXIgPSAxKTogbnVtYmVyIHtcbiAgICAgICAgY29uc3QgcmV0ID0gdGhpcy5fbmV4dFNlcXVlbnRpYWxSdW5PcmRlcjtcbiAgICAgICAgdGhpcy5fbmV4dFNlcXVlbnRpYWxSdW5PcmRlciArPSBjb3VudDtcbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0aGlzIFN0YWdlIGNvbnRhaW5zIGFuIGFjdGlvbiB0byBkZXBsb3kgdGhlIGdpdmVuIHN0YWNrLCBpZGVudGlmaWVkIGJ5IGl0cyBhcnRpZmFjdCBJRFxuICAgICAqL1xuICAgIHB1YmxpYyBkZXBsb3lzU3RhY2soYXJ0aWZhY3RJZDogc3RyaW5nKSB7XG4gICAgICAgIHJldHVybiB0aGlzLnN0YWNrc1RvRGVwbG95Lm1hcChzID0+IHMuc3RhY2tBcnRpZmFjdC5pZCkuaW5jbHVkZXMoYXJ0aWZhY3RJZCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFjdHVhbGx5IGFkZCBhbGwgdGhlIERlcGxveVN0YWNrIGFjdGlvbnMgdG8gdGhlIHN0YWdlLlxuICAgICAqXG4gICAgICogV2UgZG8gdGhpcyBsYXRlIGJlY2F1c2UgYmVmb3JlIHdlIGNhbiByZW5kZXIgdGhlIGFjdHVhbCBEZXBsb3lBY3Rpb25zLFxuICAgICAqIHdlIG5lZWQgdG8ga25vdyB3aGV0aGVyIG9yIG5vdCB3ZSBuZWVkIHRvIGNhcHR1cmUgdGhlIHN0YWNrIG91dHB1dHMuXG4gICAgICpcbiAgICAgKiBGSVhNRTogVGhpcyBpcyBoZXJlIGJlY2F1c2UgQWN0aW9ucyBhcmUgaW1tdXRhYmxlIGFuZCBjYW4ndCBiZSByZW9yZGVyZWRcbiAgICAgKiBhZnRlciBjcmVhdGlvbiwgbm9yIGlzIHRoZXJlIGEgd2F5IHRvIHNwZWNpZnkgcmVsYXRpdmUgcHJpb3JpdGllcywgd2hpY2hcbiAgICAgKiBpcyBhIGxpbWl0YXRpb24gdGhhdCB3ZSBzaG91bGQgdGFrZSBhd2F5IGluIHRoZSBiYXNlIGxpYnJhcnkuXG4gICAgICovXG4gICAgcHJpdmF0ZSBwcmVwYXJlU3RhZ2UoKSB7XG4gICAgICAgIC8vIEZJWE1FOiBNYWtlIHN1cmUgdGhpcyBvbmx5IGdldHMgcnVuIG9uY2UuIFRoZXJlIHNlZW1zIHRvIGJlIGFuIGlzc3VlIGluIHRoZSByZWNvbmNpbGlhdGlvblxuICAgICAgICAvLyBsb29wIHRoYXQgbWF5IHRyaWdnZXIgdGhpcyBtb3JlIHRoYW4gb25jZSBpZiBpdCB0aHJvd3MgYW4gZXJyb3Igc29tZXdoZXJlLCBhbmQgdGhlIGV4Y2VwdGlvblxuICAgICAgICAvLyB0aGF0IGdldHMgdGhyb3duIGhlcmUgd2lsbCB0aGVuIG92ZXJyaWRlIHRoZSBhY3R1YWwgZmFpbHVyZS5cbiAgICAgICAgaWYgKHRoaXMuX3ByZXBhcmVkKSB7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5fcHJlcGFyZWQgPSB0cnVlO1xuICAgICAgICBmb3IgKGNvbnN0IHsgcHJlcGFyZVJ1bk9yZGVyLCBzdGFja0FydGlmYWN0LCBleGVjdXRlUnVuT3JkZXIgfSBvZiB0aGlzLnN0YWNrc1RvRGVwbG95KSB7XG4gICAgICAgICAgICBjb25zdCBhcnRpZmFjdCA9IHRoaXMuaG9zdC5zdGFja091dHB1dEFydGlmYWN0KHN0YWNrQXJ0aWZhY3QuaWQpO1xuICAgICAgICAgICAgdGhpcy5waXBlbGluZVN0YWdlLmFkZEFjdGlvbihEZXBsb3lDZGtTdGFja0FjdGlvbi5mcm9tU3RhY2tBcnRpZmFjdCh0aGlzLCBzdGFja0FydGlmYWN0LCB7XG4gICAgICAgICAgICAgICAgYmFzZUFjdGlvbk5hbWU6IHRoaXMuc2ltcGxpZnlTdGFja05hbWUoc3RhY2tBcnRpZmFjdC5zdGFja05hbWUpLFxuICAgICAgICAgICAgICAgIGNsb3VkQXNzZW1ibHlJbnB1dDogdGhpcy5jbG91ZEFzc2VtYmx5QXJ0aWZhY3QsXG4gICAgICAgICAgICAgICAgb3V0cHV0OiBhcnRpZmFjdCxcbiAgICAgICAgICAgICAgICBvdXRwdXRGaWxlTmFtZTogYXJ0aWZhY3QgPyAnb3V0cHV0cy5qc29uJyA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgICAgICBwcmVwYXJlUnVuT3JkZXIsXG4gICAgICAgICAgICAgICAgZXhlY3V0ZVJ1bk9yZGVyLFxuICAgICAgICAgICAgfSkpO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkdmFuY2UgdGhlIHJ1bm9yZGVyIGNvdW50ZXIgc28gdGhhdCB0aGUgbmV4dCBzZXF1ZW50aWFsIG51bWJlciBpcyBoaWdoZXIgdGhhbiB0aGUgZ2l2ZW4gb25lXG4gICAgICovXG4gICAgcHJpdmF0ZSBhZHZhbmNlUnVuT3JkZXJQYXN0KGxhc3RVc2VkOiBudW1iZXIpIHtcbiAgICAgICAgdGhpcy5fbmV4dFNlcXVlbnRpYWxSdW5PcmRlciA9IE1hdGgubWF4KGxhc3RVc2VkICsgMSwgdGhpcy5fbmV4dFNlcXVlbnRpYWxSdW5PcmRlcik7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFNpbXBsaWZ5IHRoZSBzdGFjayBuYW1lIGJ5IHJlbW92aW5nIHRoZSBgU3RhZ2UtYCBwcmVmaXggaWYgaXQgZXhpc3RzLlxuICAgICAqL1xuICAgIHByaXZhdGUgc2ltcGxpZnlTdGFja05hbWUoczogc3RyaW5nKSB7XG4gICAgICAgIHJldHVybiBzdHJpcFByZWZpeChzLCBgJHt0aGlzLnN0YWdlTmFtZX0tYCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIE1ha2Ugc3VyZSBhbGwgYXNzZXRzIGRlcGVuZGVkIG9uIGJ5IHRoaXMgc3RhY2sgYXJlIHB1Ymxpc2hlZCBpbiB0aGlzIHBpcGVsaW5lXG4gICAgICpcbiAgICAgKiBUYWtpbmcgY2FyZSB0byBleGNsdWRlIHRoZSBzdGFjayB0ZW1wbGF0ZSBpdHNlbGYgLS0gaXQgaXMgYmVpbmcgcHVibGlzaGVkXG4gICAgICogYXMgYW4gYXNzZXQgYmVjYXVzZSB0aGUgQ0xJIG5lZWRzIHRvIGtub3cgdGhlIGFzc2V0IHB1Ymxpc2hpbmcgcm9sZSB3aGVuXG4gICAgICogcHVzaGluZyB0aGUgdGVtcGxhdGUgdG8gUzMsIGJ1dCBpbiB0aGUgY2FzZSBvZiBDb2RlUGlwZWxpbmUgd2UgYWx3YXlzXG4gICAgICogcmVmZXJlbmNlIHRoZSB0ZW1wbGF0ZSBmcm9tIHRoZSBhcnRpZmFjdCBidWNrZXQuXG4gICAgICpcbiAgICAgKiAoTk9URTogdGhpcyBpcyBvbmx5IHRydWUgZm9yIHRvcC1sZXZlbCBzdGFja3MsIG5vdCBuZXN0ZWQgc3RhY2tzLiBOZXN0ZWRcbiAgICAgKiBTdGFjayB0ZW1wbGF0ZXMgYXJlIGFsd2F5cyBwdWJsaXNoZWQgYXMgYXNzZXRzKS5cbiAgICAgKi9cbiAgICBwcml2YXRlIHB1Ymxpc2hBc3NldERlcGVuZGVuY2llcyhzdGFja0FydGlmYWN0OiBjeGFwaS5DbG91ZEZvcm1hdGlvblN0YWNrQXJ0aWZhY3QpIHtcbiAgICAgICAgY29uc3QgYXNzZXRNYW5pZmVzdHMgPSBzdGFja0FydGlmYWN0LmRlcGVuZGVuY2llcy5maWx0ZXIoaXNBc3NldE1hbmlmZXN0KTtcbiAgICAgICAgZm9yIChjb25zdCBtYW5pZmVzdEFydGlmYWN0IG9mIGFzc2V0TWFuaWZlc3RzKSB7XG4gICAgICAgICAgICBjb25zdCBtYW5pZmVzdCA9IEFzc2V0TWFuaWZlc3RSZWFkZXIuZnJvbUZpbGUobWFuaWZlc3RBcnRpZmFjdC5maWxlKTtcbiAgICAgICAgICAgIGZvciAoY29uc3QgZW50cnkgb2YgbWFuaWZlc3QuZW50cmllcykge1xuICAgICAgICAgICAgICAgIGxldCBhc3NldFR5cGU6IEFzc2V0VHlwZTtcbiAgICAgICAgICAgICAgICBpZiAoZW50cnkgaW5zdGFuY2VvZiBEb2NrZXJJbWFnZU1hbmlmZXN0RW50cnkpIHtcbiAgICAgICAgICAgICAgICAgICAgYXNzZXRUeXBlID0gQXNzZXRUeXBlLkRPQ0tFUl9JTUFHRTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgZWxzZSBpZiAoZW50cnkgaW5zdGFuY2VvZiBGaWxlTWFuaWZlc3RFbnRyeSkge1xuICAgICAgICAgICAgICAgICAgICAvLyBEb24ndCBwdWJsaXNoZyB0aGUgdGVtcGxhdGUgZm9yIHRoaXMgc3RhY2tcbiAgICAgICAgICAgICAgICAgICAgaWYgKGVudHJ5LnNvdXJjZS5wYWNrYWdpbmcgPT09ICdmaWxlJyAmJiBlbnRyeS5zb3VyY2UucGF0aCA9PT0gc3RhY2tBcnRpZmFjdC50ZW1wbGF0ZUZpbGUpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGNvbnRpbnVlO1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIGFzc2V0VHlwZSA9IEFzc2V0VHlwZS5GSUxFO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnJlY29nbml6ZWQgYXNzZXQgdHlwZTogJHtlbnRyeS50eXBlfWApO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICB0aGlzLmhvc3QucHVibGlzaEFzc2V0KHtcbiAgICAgICAgICAgICAgICAgICAgYXNzZXRNYW5pZmVzdFBhdGg6IG1hbmlmZXN0QXJ0aWZhY3QuZmlsZSxcbiAgICAgICAgICAgICAgICAgICAgYXNzZXRJZDogZW50cnkuaWQuYXNzZXRJZCxcbiAgICAgICAgICAgICAgICAgICAgYXNzZXRTZWxlY3RvcjogZW50cnkuaWQudG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICAgICAgYXNzZXRUeXBlLFxuICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgfVxufVxuLyoqXG4gKiBBZGRpdGlvbmFsIG9wdGlvbnMgZm9yIGFkZGluZyBhIHN0YWNrIGRlcGxveW1lbnRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBZGRTdGFja09wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIEJhc2UgcnVub3JkZXJcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTmV4dCBzZXF1ZW50aWFsIHJ1bm9yZGVyXG4gICAgICovXG4gICAgcmVhZG9ubHkgcnVuT3JkZXI/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogQmFzZSBydW5vcmRlclxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBydW5PcmRlciArIDFcbiAgICAgKi9cbiAgICByZWFkb25seSBleGVjdXRlUnVuT3JkZXI/OiBudW1iZXI7XG59XG4vKipcbiAqIEEgc2luZ2xlIG91dHB1dCBvZiBhIFN0YWNrXG4gKi9cbmV4cG9ydCBjbGFzcyBTdGFja091dHB1dCB7XG4gICAgLyoqXG4gICAgICogVGhlIGFydGlmYWN0IGFuZCBmaWxlIHRoZSBvdXRwdXQgaXMgc3RvcmVkIGluXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGFydGlmYWN0RmlsZTogY29kZXBpcGVsaW5lLkFydGlmYWN0UGF0aDtcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgb3V0cHV0IGluIHRoZSBKU09OIG9iamVjdCBpbiB0aGUgZmlsZVxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBvdXRwdXROYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogQnVpbGQgYSBTdGFja091dHB1dCBmcm9tIGEga25vd24gYXJ0aWZhY3QgYW5kIGFuIG91dHB1dCBuYW1lXG4gICAgICovXG4gICAgY29uc3RydWN0b3IoYXJ0aWZhY3RGaWxlOiBjb2RlcGlwZWxpbmUuQXJ0aWZhY3RQYXRoLCBvdXRwdXROYW1lOiBzdHJpbmcpIHtcbiAgICAgICAgdGhpcy5hcnRpZmFjdEZpbGUgPSBhcnRpZmFjdEZpbGU7XG4gICAgICAgIHRoaXMub3V0cHV0TmFtZSA9IG91dHB1dE5hbWU7XG4gICAgfVxufVxuZnVuY3Rpb24gc3RyaXBQcmVmaXgoczogc3RyaW5nLCBwcmVmaXg6IHN0cmluZykge1xuICAgIHJldHVybiBzLnN0YXJ0c1dpdGgocHJlZml4KSA/IHMuc3Vic3RyKHByZWZpeC5sZW5ndGgpIDogcztcbn1cbmZ1bmN0aW9uIGlzQXNzZXRNYW5pZmVzdChzOiBjeGFwaS5DbG91ZEFydGlmYWN0KTogcyBpcyBjeGFwaS5Bc3NldE1hbmlmZXN0QXJ0aWZhY3Qge1xuICAgIC8vIGluc3RhbmNlb2YgaXMgdG9vIHJpc2t5LCBhbmQgd2UncmUgYXQgYSB0b28gbGF0ZSBzdGFnZSB0byBwcm9wZXJseSBmaXguXG4gICAgLy8gcmV0dXJuIHMgaW5zdGFuY2VvZiBjeGFwaS5Bc3NldE1hbmlmZXN0QXJ0aWZhY3Q7XG4gICAgcmV0dXJuIHMuY29uc3RydWN0b3IubmFtZSA9PT0gJ0Fzc2V0TWFuaWZlc3RBcnRpZmFjdCc7XG59XG4vKipcbiAqIEZlYXR1cmVzIHRoYXQgdGhlIFN0YWdlIG5lZWRzIGZyb20gaXRzIGVudmlyb25tZW50XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVN0YWdlSG9zdCB7XG4gICAgLyoqXG4gICAgICogTWFrZSBzdXJlIGFsbCB0aGUgYXNzZXRzIGZyb20gdGhlIGdpdmVuIG1hbmlmZXN0IGFyZSBwdWJsaXNoZWRcbiAgICAgKi9cbiAgICBwdWJsaXNoQXNzZXQoY29tbWFuZDogQXNzZXRQdWJsaXNoaW5nQ29tbWFuZCk6IHZvaWQ7XG4gICAgLyoqXG4gICAgICogUmV0dXJuIHRoZSBBcnRpZmFjdCB0aGUgZ2l2ZW4gc3RhY2sgaGFzIHRvIGVtaXQgaXRzIG91dHB1dHMgaW50bywgaWYgYW55XG4gICAgICovXG4gICAgc3RhY2tPdXRwdXRBcnRpZmFjdChzdGFja0FydGlmYWN0SWQ6IHN0cmluZyk6IGNvZGVwaXBlbGluZS5BcnRpZmFjdCB8IHVuZGVmaW5lZDtcbn1cbi8qKlxuICogSW5zdHJ1Y3Rpb25zIHRvIHB1Ymxpc2ggY2VydGFpbiBhc3NldHNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBc3NldFB1Ymxpc2hpbmdDb21tYW5kIHtcbiAgICAvKipcbiAgICAgKiBBc3NldCBtYW5pZmVzdCBwYXRoXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXNzZXRNYW5pZmVzdFBhdGg6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBBc3NldCBpZGVudGlmaWVyXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXNzZXRJZDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEFzc2V0IHNlbGVjdG9yIHRvIHBhc3MgdG8gYGNkay1hc3NldHNgLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFzc2V0U2VsZWN0b3I6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUeXBlIG9mIGFzc2V0IHRvIHB1Ymxpc2hcbiAgICAgKi9cbiAgICByZWFkb25seSBhc3NldFR5cGU6IEFzc2V0VHlwZTtcbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgYWRkaW5nIGFuIGFwcGxpY2F0aW9uIHN0YWdlIHRvIGEgcGlwZWxpbmVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBZGRTdGFnZU9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIEFkZCBtYW51YWwgYXBwcm92YWxzIGJlZm9yZSBleGVjdXRpbmcgY2hhbmdlIHNldHNcbiAgICAgKlxuICAgICAqIFRoaXMgZ2l2ZXMgaHVtYW5zIHRoZSBvcHBvcnR1bml0eSB0byBjb25maXJtIHRoZSBjaGFuZ2Ugc2V0IGxvb2tzIGFscmlnaHRcbiAgICAgKiBiZWZvcmUgZGVwbG95aW5nIGl0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAgKi9cbiAgICByZWFkb25seSBtYW51YWxBcHByb3ZhbHM/OiBib29sZWFuO1xufVxuLyoqXG4gKiBPcHRpb25zIGZvciBhZGRNYW51YWxBcHByb3ZhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFkZE1hbnVhbEFwcHJvdmFsT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIG1hbnVhbCBhcHByb3ZhbCBhY3Rpb25cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0ICdNYW51YWxBcHByb3ZhbCcgd2l0aCBhIHJvbGxpbmcgY291bnRlclxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFjdGlvbk5hbWU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHJ1bk9yZGVyIGZvciB0aGlzIGFjdGlvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBUaGUgbmV4dCBzZXF1ZW50aWFsIHJ1bk9yZGVyXG4gICAgICovXG4gICAgcmVhZG9ubHkgcnVuT3JkZXI/OiBudW1iZXI7XG59XG4vKipcbiAqIFF1ZXVlZCBcImRlcGxveSBzdGFja1wiIGNvbW1hbmQgdGhhdCBpcyByZWlmaWVkIGR1cmluZyBwcmVwYXJlKClcbiAqL1xuaW50ZXJmYWNlIERlcGxveVN0YWNrQ29tbWFuZCB7XG4gICAgcHJlcGFyZVJ1bk9yZGVyOiBudW1iZXI7XG4gICAgZXhlY3V0ZVJ1bk9yZGVyOiBudW1iZXI7XG4gICAgc3RhY2tBcnRpZmFjdDogY3hhcGkuQ2xvdWRGb3JtYXRpb25TdGFja0FydGlmYWN0O1xufVxuIl19