"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const path = require("path");
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 */
class CdkPipeline extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
            ...props,
            restartExecutionOnUpdate: true,
            stages: [
                {
                    stageName: 'Source',
                    actions: [props.sourceAction],
                },
                {
                    stageName: 'Build',
                    actions: [props.synthAction],
                },
                {
                    stageName: 'UpdatePipeline',
                    actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                            cloudAssemblyInput: this._cloudAssemblyArtifact,
                            pipelineStackName: pipelineStack.stackName,
                            cdkCliVersion: props.cdkCliVersion,
                            projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                        })],
                },
            ],
        });
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
        });
        core_1.Aspects.of(this).add({ visit: () => this._assets.removeAssetsStageIfEmpty() });
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    this.node.addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.publishers = {};
        this.assetRoles = {};
        this._fileAssetCtr = 1;
        this._dockerAssetCtr = 1;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        // We MUST add the Stage immediately here, otherwise it will be in the wrong place
        // in the pipeline!
        this.stage = this.props.pipeline.addStage({ stageName: 'Assets' });
        this.pipeline = this.props.pipeline;
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${this._fileAssetCtr++}` : `DockerAsset${this._dockerAssetCtr++}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
            });
            this.stage.addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * Remove the Assets stage if it turns out we didn't add any Assets to publish
     */
    removeAssetsStageIfEmpty() {
        if (Object.keys(this.publishers).length === 0) {
            // Hacks to get access to innards of Pipeline
            // Modify 'stages' array in-place to remove Assets stage if empty
            const stages = this.props.pipeline._stages;
            const ix = stages.indexOf(this.stage);
            if (ix > -1) {
                stages.splice(ix, 1);
            }
        }
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === actions_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new iam.Role(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Logging permissions
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: '/aws/codebuild/*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: [logGroupArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        const codeBuildArn = core_1.Stack.of(this).formatArn({
            service: 'codebuild',
            resource: 'report-group',
            resourceName: '*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
            ],
            resources: [codeBuildArn],
        }));
        // CodeBuild start/stop
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // Publishing role access
        const rolePattern = assetType === actions_1.AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        // Artifact access
        this.pipeline.artifactBucket.grantRead(assetRole);
        this.assetRoles[assetType] = assetRole.withoutPolicyUpdates();
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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