"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployCdkStackAction = void 0;
const path = require("path");
const cfn = require("../../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const cpactions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const construct_internals_1 = require("../private/construct-internals");
/**
 * Action to deploy a CDK Stack
 *
 * Adds two CodePipeline Actions to the pipeline: one to create a ChangeSet
 * and one to execute it.
 *
 * You do not need to instantiate this action yourself -- it will automatically
 * be added by the pipeline when you add stack artifacts or entire stages.
 */
class DeployCdkStackAction {
    constructor(props) {
        var _a, _b, _c, _d, _e;
        if (props.output && !props.outputFileName) {
            throw new Error('If \'output\' is set, \'outputFileName\' is also required');
        }
        this.stackArtifactId = props.stackArtifactId;
        this.dependencyStackArtifactIds = (_a = props.dependencyStackArtifactIds) !== null && _a !== void 0 ? _a : [];
        this.prepareRunOrder = (_b = props.prepareRunOrder) !== null && _b !== void 0 ? _b : 1;
        this.executeRunOrder = (_c = props.executeRunOrder) !== null && _c !== void 0 ? _c : this.prepareRunOrder + 1;
        this.stackName = props.stackName;
        const baseActionName = (_d = props.baseActionName) !== null && _d !== void 0 ? _d : this.stackName;
        const changeSetName = (_e = props.changeSetName) !== null && _e !== void 0 ? _e : 'PipelineChange';
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: `${baseActionName}.Prepare`,
            changeSetName,
            runOrder: this.prepareRunOrder,
            stackName: this.stackName,
            templatePath: props.cloudAssemblyInput.atPath(props.templatePath),
            adminPermissions: false,
            role: props.actionRole,
            deploymentRole: props.cloudFormationExecutionRole,
            region: props.region,
            capabilities: [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND],
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: `${baseActionName}.Deploy`,
            changeSetName,
            runOrder: this.executeRunOrder,
            stackName: this.stackName,
            role: props.actionRole,
            region: props.region,
            outputFileName: props.outputFileName,
            output: props.output,
        });
    }
    /**
     * Construct a DeployCdkStackAction from a Stack artifact
     */
    static fromStackArtifact(scope, artifact, options) {
        var _a;
        if (!artifact.assumeRoleArn) {
            throw new Error(`Stack '${artifact.stackName}' does not have deployment role information; use the 'DefaultStackSynthesizer' synthesizer, or set the '@aws-cdk/core:newStyleStackSynthesis' context key.`);
        }
        const artRegion = artifact.environment.region;
        const region = artRegion === core_1.Stack.of(scope).region || artRegion === cxapi.UNKNOWN_REGION ? undefined : artRegion;
        const artAccount = artifact.environment.account;
        const account = artAccount === core_1.Stack.of(scope).account || artAccount === cxapi.UNKNOWN_ACCOUNT ? undefined : artAccount;
        const actionRole = roleFromPlaceholderArn(scope, region, account, artifact.assumeRoleArn);
        const cloudFormationExecutionRole = roleFromPlaceholderArn(scope, region, account, artifact.cloudFormationExecutionRoleArn);
        // We need the path of the template relative to the root Cloud Assembly
        // It should be easier to get this, but for now it is what it is.
        const appAsmRoot = construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(scope)).outdir;
        const fullTemplatePath = path.join(artifact.assembly.directory, artifact.templateFile);
        const templatePath = path.relative(appAsmRoot, fullTemplatePath);
        return new DeployCdkStackAction({
            actionRole,
            cloudFormationExecutionRole,
            templatePath,
            region,
            stackArtifactId: artifact.id,
            dependencyStackArtifactIds: artifact.dependencies.filter(isStackArtifact).map(s => s.id),
            stackName: (_a = options.stackName) !== null && _a !== void 0 ? _a : artifact.stackName,
            ...options,
        });
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        stage.addAction(this.prepareChangeSetAction);
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.DeployCdkStackAction = DeployCdkStackAction;
function roleFromPlaceholderArn(scope, region, account, arn) {
    if (!arn) {
        return undefined;
    }
    // Use placeholdered arn as construct ID.
    const id = arn;
    // https://github.com/aws/aws-cdk/issues/7255
    let existingRole = scope.node.tryFindChild(`ImmutableRole${id}`);
    if (existingRole) {
        return existingRole;
    }
    // For when #7255 is fixed.
    existingRole = scope.node.tryFindChild(id);
    if (existingRole) {
        return existingRole;
    }
    const arnToImport = cxapi.EnvironmentPlaceholders.replace(arn, {
        region: region !== null && region !== void 0 ? region : core_1.Aws.REGION,
        accountId: account !== null && account !== void 0 ? account : core_1.Aws.ACCOUNT_ID,
        partition: core_1.Aws.PARTITION,
    });
    return iam.Role.fromRoleArn(scope, id, arnToImport, { mutable: false });
}
function isStackArtifact(a) {
    // instanceof is too risky, and we're at a too late stage to properly fix.
    // return a instanceof cxapi.CloudFormationStackArtifact;
    return a.constructor.name === 'CloudFormationStackArtifact';
}
//# sourceMappingURL=data:application/json;base64,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