"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.assertBound = exports.contentHash = exports.addStackArtifactToAssembly = void 0;
const crypto = require("crypto");
const cxschema = require("../../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
/**
 * Shared logic of writing stack artifact to the Cloud Assembly
 *
 * This logic is shared between StackSyntheses.
 *
 * It could have been a protected method on a base class, but it
 * uses `Partial<cxapi.AwsCloudFormationStackProperties>` in the
 * parameters (which is convenient so I can remain typesafe without
 * copy/pasting), and jsii will choke on this type.
 */
function addStackArtifactToAssembly(session, stack, stackProps, additionalStackDependencies) {
    // nested stack tags are applied at the AWS::CloudFormation::Stack resource
    // level and are not needed in the cloud assembly.
    // TODO: move these to the cloud assembly artifact properties instead of metadata
    if (stack.tags.hasTags()) {
        stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.STACK_TAGS, stack.tags.renderTags());
    }
    const deps = [
        ...stack.dependencies.map(s => s.artifactId),
        ...additionalStackDependencies,
    ];
    const meta = collectStackMetadata(stack);
    // backwards compatibility since originally artifact ID was always equal to
    // stack name the stackName attribute is optional and if it is not specified
    // the CLI will use the artifact ID as the stack name. we *could have*
    // always put the stack name here but wanted to minimize the risk around
    // changes to the assembly manifest. so this means that as long as stack
    // name and artifact ID are the same, the cloud assembly manifest will not
    // change.
    const stackNameProperty = stack.stackName === stack.artifactId
        ? {}
        : { stackName: stack.stackName };
    const properties = {
        templateFile: stack.templateFile,
        terminationProtection: stack.terminationProtection,
        ...stackProps,
        ...stackNameProperty,
    };
    // add an artifact that represents this stack
    session.assembly.addArtifact(stack.artifactId, {
        type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: stack.environment,
        properties,
        dependencies: deps.length > 0 ? deps : undefined,
        metadata: Object.keys(meta).length > 0 ? meta : undefined,
    });
}
exports.addStackArtifactToAssembly = addStackArtifactToAssembly;
/**
 * Collect the metadata from a stack
 */
function collectStackMetadata(stack) {
    const output = {};
    visit(stack);
    return output;
    function visit(node) {
        // break off if we reached a node that is not a child of this stack
        const parent = findParentStack(node);
        if (parent !== stack) {
            return;
        }
        if (node.node.metadata.length > 0) {
            // Make the path absolute
            output[construct_compat_1.ConstructNode.PATH_SEP + node.node.path] = node.node.metadata.map(md => stack.resolve(md));
        }
        for (const child of node.node.children) {
            visit(child);
        }
    }
    function findParentStack(node) {
        if (node instanceof stack_1.Stack && node.nestedStackParent === undefined) {
            return node;
        }
        if (!node.node.scope) {
            return undefined;
        }
        return findParentStack(node.node.scope);
    }
}
/**
 * Hash a string
 */
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
exports.contentHash = contentHash;
/**
 * Throw an error message about binding() if we don't have a value for x.
 *
 * This replaces the ! assertions we would need everywhere otherwise.
 */
function assertBound(x) {
    if (x === null && x === undefined) {
        throw new Error('You must call bindStack() first');
    }
}
exports.assertBound = assertBound;
//# sourceMappingURL=data:application/json;base64,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