"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SizeRoundingBehavior = exports.Size = void 0;
const token_1 = require("./token");
/**
 * Represents the amount of digital storage.
 *
 * The amount can be specified either as a literal value (e.g: `10`) which
 * cannot be negative, or as an unresolved number token.
 *
 * When the amount is passed as a token, unit conversion is not possible.
 */
class Size {
    constructor(amount, unit) {
        if (!token_1.Token.isUnresolved(amount) && amount < 0) {
            throw new Error(`Storage amounts cannot be negative. Received: ${amount}`);
        }
        this.amount = amount;
        this.unit = unit;
    }
    /**
     * Create a Storage representing an amount kibibytes.
     * 1 KiB = 1024 bytes
     */
    static kibibytes(amount) {
        return new Size(amount, StorageUnit.Kibibytes);
    }
    /**
     * Create a Storage representing an amount mebibytes.
     * 1 MiB = 1024 KiB
     */
    static mebibytes(amount) {
        return new Size(amount, StorageUnit.Mebibytes);
    }
    /**
     * Create a Storage representing an amount gibibytes.
     * 1 GiB = 1024 MiB
     */
    static gibibytes(amount) {
        return new Size(amount, StorageUnit.Gibibytes);
    }
    /**
     * Create a Storage representing an amount tebibytes.
     * 1 TiB = 1024 GiB
     */
    static tebibytes(amount) {
        return new Size(amount, StorageUnit.Tebibytes);
    }
    /**
     * Create a Storage representing an amount pebibytes.
     * 1 PiB = 1024 TiB
     */
    static pebibyte(amount) {
        return new Size(amount, StorageUnit.Pebibytes);
    }
    /**
     * Return this storage as a total number of kibibytes.
     */
    toKibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Kibibytes, opts);
    }
    /**
     * Return this storage as a total number of mebibytes.
     */
    toMebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Mebibytes, opts);
    }
    /**
     * Return this storage as a total number of gibibytes.
     */
    toGibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Gibibytes, opts);
    }
    /**
     * Return this storage as a total number of tebibytes.
     */
    toTebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Tebibytes, opts);
    }
    /**
     * Return this storage as a total number of pebibytes.
     */
    toPebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Pebibytes, opts);
    }
}
exports.Size = Size;
/**
 * Rounding behaviour when converting between units of `Size`.
 */
var SizeRoundingBehavior;
(function (SizeRoundingBehavior) {
    /** Fail the conversion if the result is not an integer. */
    SizeRoundingBehavior[SizeRoundingBehavior["FAIL"] = 0] = "FAIL";
    /** If the result is not an integer, round it to the closest integer less than the result */
    SizeRoundingBehavior[SizeRoundingBehavior["FLOOR"] = 1] = "FLOOR";
    /** Don't round. Return even if the result is a fraction. */
    SizeRoundingBehavior[SizeRoundingBehavior["NONE"] = 2] = "NONE";
})(SizeRoundingBehavior = exports.SizeRoundingBehavior || (exports.SizeRoundingBehavior = {}));
class StorageUnit {
    constructor(label, inKibiBytes) {
        this.label = label;
        this.inKibiBytes = inKibiBytes;
        // MAX_SAFE_INTEGER is 2^53, so by representing storage in kibibytes,
        // the highest storage we can represent is 8 exbibytes.
    }
    toString() {
        return this.label;
    }
}
StorageUnit.Kibibytes = new StorageUnit('kibibytes', 1);
StorageUnit.Mebibytes = new StorageUnit('mebibytes', 1024);
StorageUnit.Gibibytes = new StorageUnit('gibibytes', 1024 * 1024);
StorageUnit.Tebibytes = new StorageUnit('tebibytes', 1024 * 1024 * 1024);
StorageUnit.Pebibytes = new StorageUnit('pebibytes', 1024 * 1024 * 1024 * 1024);
function convert(amount, fromUnit, toUnit, options = {}) {
    var _a;
    const rounding = (_a = options.rounding) !== null && _a !== void 0 ? _a : SizeRoundingBehavior.FAIL;
    if (fromUnit.inKibiBytes === toUnit.inKibiBytes) {
        return amount;
    }
    if (token_1.Token.isUnresolved(amount)) {
        throw new Error(`Unable to perform time unit conversion on un-resolved token ${amount}.`);
    }
    const multiplier = fromUnit.inKibiBytes / toUnit.inKibiBytes;
    const value = amount * multiplier;
    switch (rounding) {
        case SizeRoundingBehavior.NONE:
            return value;
        case SizeRoundingBehavior.FLOOR:
            return Math.floor(value);
        default:
        case SizeRoundingBehavior.FAIL:
            if (!Number.isInteger(value)) {
                throw new Error(`'${amount} ${fromUnit}' cannot be converted into a whole number of ${toUnit}.`);
            }
            return value;
    }
}
//# sourceMappingURL=data:application/json;base64,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