"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConstructNode = exports.ConstructOrder = exports.Construct = void 0;
/**
 * Constructs compatibility layer.
 *
 * This file includes types that shadow types in the "constructs" module in
 * order to allow backwards-compatiblity in the AWS CDK v1.0 release line.
 *
 * There are pretty ugly hacks here, which mostly involve downcasting types to
 * adhere to legacy AWS CDK APIs.
 *
 * This file, in its entirety, is expected to be removed in v2.0.
 */
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const constructs = require("constructs");
const annotations_1 = require("./annotations");
const aspect_1 = require("./aspect");
const token_1 = require("./token");
const ORIGINAL_CONSTRUCT_NODE_SYMBOL = Symbol.for('@aws-cdk/core.ConstructNode');
const CONSTRUCT_SYMBOL = Symbol.for('@aws-cdk/core.Construct');
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 */
class Construct extends constructs.Construct {
    constructor(scope, id) {
        super(scope, id, {
            nodeFactory: {
                createNode: (h, s, i) => new ConstructNode(h, s, i)._actualNode,
            },
        });
        if (token_1.Token.isUnresolved(id)) {
            throw new Error(`Cannot use tokens in construct ID: ${id}`);
        }
        Object.defineProperty(this, CONSTRUCT_SYMBOL, { value: true });
        this.node = ConstructNode._unwrap(constructs.Node.of(this));
        const disableTrace = this.node.tryGetContext(cxapi.DISABLE_METADATA_STACK_TRACE) ||
            this.node.tryGetContext(constructs.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA) ||
            process.env.CDK_DISABLE_STACK_TRACE;
        if (disableTrace) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
            this.node.setContext(constructs.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA, true);
            process.env.CDK_DISABLE_STACK_TRACE = '1';
        }
    }
    /**
     * Return whether the given object is a Construct
     */
    static isConstruct(x) {
        return typeof x === 'object' && x !== null && CONSTRUCT_SYMBOL in x;
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if the construct is valid.
     */
    onValidate() {
        return this.validate();
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    onPrepare() {
        this.prepare();
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    onSynthesize(session) {
        this.synthesize({
            outdir: session.outdir,
            assembly: session.assembly,
        });
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if the construct is valid.
     */
    validate() {
        return [];
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    prepare() {
        return;
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    synthesize(session) {
        ignore(session);
    }
}
exports.Construct = Construct;
/**
 * In what order to return constructs
 */
var ConstructOrder;
(function (ConstructOrder) {
    /**
     * Depth-first, pre-order
     */
    ConstructOrder[ConstructOrder["PREORDER"] = 0] = "PREORDER";
    /**
     * Depth-first, post-order (leaf nodes first)
     */
    ConstructOrder[ConstructOrder["POSTORDER"] = 1] = "POSTORDER";
})(ConstructOrder = exports.ConstructOrder || (exports.ConstructOrder = {}));
/**
 * Represents the construct node in the scope tree.
 */
class ConstructNode {
    constructor(host, scope, id) {
        this.host = host;
        this._actualNode = new constructs.Node(host, scope, id);
        // store a back reference on _actualNode so we can our ConstructNode from it
        Object.defineProperty(this._actualNode, ORIGINAL_CONSTRUCT_NODE_SYMBOL, {
            value: this,
            configurable: false,
            enumerable: false,
        });
    }
    /**
     * Returns the wrapping `@aws-cdk/core.ConstructNode` instance from a `constructs.ConstructNode`.
     *
     * @internal
     */
    static _unwrap(c) {
        const x = c[ORIGINAL_CONSTRUCT_NODE_SYMBOL];
        if (!x) {
            throw new Error('invalid ConstructNode type');
        }
        return x;
    }
    /**
     * Synthesizes a CloudAssembly from a construct tree.
     * @param node The root of the construct tree.
     * @param options Synthesis options.
     * @deprecated Use `app.synth()` or `stage.synth()` instead
     */
    static synth(node, options = {}) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const a = require('./private/synthesis');
        return a.synthesize(node.root, options);
    }
    /**
     * Invokes "prepare" on all constructs (depth-first, post-order) in the tree under `node`.
     * @param node The root node
     * @deprecated Use `app.synth()` instead
     */
    static prepare(node) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const p = require('./private/prepare-app');
        p.prepareApp(node.root); // resolve cross refs and nested stack assets.
        return node._actualNode.prepare();
    }
    /**
     * Invokes "validate" on all constructs in the tree (depth-first, pre-order) and returns
     * the list of all errors. An empty list indicates that there are no errors.
     *
     * @param node The root node
     */
    static validate(node) {
        return node._actualNode.validate().map(e => ({ source: e.source, message: e.message }));
    }
    /**
     * Returns the scope in which this construct is defined.
     *
     * The value is `undefined` at the root of the construct scope tree.
     */
    get scope() {
        return this._actualNode.scope;
    }
    /**
     * The id of this construct within the current scope.
     *
     * This is a a scope-unique id. To obtain an app-unique id for this construct, use `uniqueId`.
     */
    get id() { return this._actualNode.id; }
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     */
    get path() { return this._actualNode.path; }
    /**
     * A tree-global unique alphanumeric identifier for this construct.
     * Includes all components of the tree.
     */
    get uniqueId() { return this._actualNode.uniqueId; }
    /**
     * Return a direct child by id, or undefined
     *
     * @param id Identifier of direct child
     * @returns the child if found, or undefined
     */
    tryFindChild(id) { return this._actualNode.tryFindChild(id); }
    /**
     * Return a direct child by id
     *
     * Throws an error if the child is not found.
     *
     * @param id Identifier of direct child
     * @returns Child with the given id.
     */
    findChild(id) { return this._actualNode.findChild(id); }
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     *
     * @throws if there is more than one child
     * @returns a construct or undefined if there is no default child
     */
    get defaultChild() { return this._actualNode.defaultChild; }
    /**
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     */
    set defaultChild(value) { this._actualNode.defaultChild = value; }
    /**
     * All direct children of this construct.
     */
    get children() { return this._actualNode.children; }
    /**
     * Return this construct and all of its children in the given order
     */
    findAll(order = ConstructOrder.PREORDER) { return this._actualNode.findAll(order); }
    /**
     * This can be used to set contextual values.
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     * @param key The context key
     * @param value The context value
     */
    setContext(key, value) {
        if (token_1.Token.isUnresolved(key)) {
            throw new Error('Invalid context key: context keys can\'t include tokens');
        }
        this._actualNode.setContext(key, value);
    }
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key
     * @returns The context value or `undefined` if there is no context value for thie key.
     */
    tryGetContext(key) {
        if (token_1.Token.isUnresolved(key)) {
            throw new Error('Invalid context key: context keys can\'t include tokens');
        }
        return this._actualNode.tryGetContext(key);
    }
    /**
     * An immutable array of metadata objects associated with this construct.
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     */
    get metadata() { return this._actualNode.metadata; }
    /**
     * Adds a metadata entry to this construct.
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata
     * @param data the value of the metadata (can be a Token). If null/undefined, metadata will not be added.
     * @param fromFunction a function under which to restrict the metadata entry's stack trace (defaults to this.addMetadata)
     */
    addMetadata(type, data, fromFunction) { this._actualNode.addMetadata(type, data, fromFunction); }
    /**
     * DEPRECATED: Adds a { "info": <message> } metadata entry to this construct.
     * The toolkit will display the info message when apps are synthesized.
     * @param message The info message.
     * @deprecated use `Annotations.of(construct).addInfo()`
     */
    addInfo(message) {
        annotations_1.Annotations.of(this.host).addInfo(message);
    }
    /**
     * DEPRECATED: Adds a { "warning": <message> } metadata entry to this construct.
     * The toolkit will display the warning when an app is synthesized, or fail
     * if run in --strict mode.
     * @param message The warning message.
     * @deprecated use `Annotations.of(construct).addWarning()`
     */
    addWarning(message) {
        annotations_1.Annotations.of(this.host).addWarning(message);
    }
    /**
     * DEPRECATED: Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     * @deprecated use `Annotations.of(construct).addError()`
     */
    addError(message) {
        annotations_1.Annotations.of(this.host).addError(message);
    }
    /**
     * DEPRECATED: Applies the aspect to this Constructs node
     *
     * @deprecated This API is going to be removed in the next major version of
     * the AWS CDK. Please use `Aspects.of(scope).add()` instead.
     */
    applyAspect(aspect) {
        annotations_1.Annotations.of(this.host).addDeprecation('@aws-cdk/core.ConstructNode.applyAspect', 'Use "Aspects.of(construct).add(aspect)" instead');
        aspect_1.Aspects.of(this.host).add(aspect);
    }
    /**
     * All parent scopes of this construct.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be the current construct and the first element will be the root of the
     * tree.
     */
    get scopes() { return this._actualNode.scopes; }
    /**
     * @returns The root of the construct tree.
     */
    get root() { return this._actualNode.root; }
    /**
     * Returns true if this construct or the scopes in which it is defined are
     * locked.
     */
    get locked() { return this._actualNode.locked; }
    /**
     * Add an ordering dependency on another Construct.
     *
     * All constructs in the dependency's scope will be deployed before any
     * construct in this construct's scope.
     */
    addDependency(...dependencies) { this._actualNode.addDependency(...dependencies); }
    /**
     * Return all dependencies registered on this node or any of its children
     */
    get dependencies() { return this._actualNode.dependencies; }
    /**
     * Remove the child with the given name, if present.
     *
     * @returns Whether a child with the given name was deleted.
     * @experimental
     */
    tryRemoveChild(childName) { return this._actualNode.tryRemoveChild(childName); }
}
exports.ConstructNode = ConstructNode;
/**
 * Separator used to delimit construct path components.
 */
ConstructNode.PATH_SEP = '/';
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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