"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachineFragment = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const chain_1 = require("./chain");
const parallel_1 = require("./states/parallel");
const state_1 = require("./states/state");
/**
 * Base class for reusable state machine fragments
 */
class StateMachineFragment extends cdk.Construct {
    get id() {
        return this.node.id;
    }
    /**
     * Prefix the IDs of all states in this state machine fragment
     *
     * Use this to avoid multiple copies of the state machine all having the
     * same state IDs.
     *
     * @param prefix The prefix to add. Will use construct ID by default.
     */
    prefixStates(prefix) {
        state_1.State.prefixStates(this, prefix || `${this.id}: `);
        return this;
    }
    /**
     * Wrap all states in this state machine fragment up into a single state.
     *
     * This can be used to add retry or error handling onto this state
     * machine fragment.
     *
     * Be aware that this changes the result of the inner state machine
     * to be an array with the result of the state machine in it. Adjust
     * your paths accordingly. For example, change 'outputPath' to
     * '$[0]'.
     */
    toSingleState(options = {}) {
        const stateId = options.stateId || this.id;
        this.prefixStates(options.prefixStates || `${stateId}: `);
        return new parallel_1.Parallel(this, stateId, options).branch(this);
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        return chain_1.Chain.start(this).next(next);
    }
}
exports.StateMachineFragment = StateMachineFragment;
//# sourceMappingURL=data:application/json;base64,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