"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputType = exports.TaskInput = void 0;
const fields_1 = require("./fields");
/**
 * Type union for task classes that accept multiple types of payload
 */
class TaskInput {
    /**
     *
     * @param type type of task input
     * @param value payload for the corresponding input type.
     * It can be a JSON-encoded object, context, data, etc.
     */
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    /**
     * Use a literal string as task input
     *
     * This might be a JSON-encoded object, or just a text.
     */
    static fromText(text) {
        return new TaskInput(InputType.TEXT, text);
    }
    /**
     * Use an object as task input
     *
     * This object may contain Data and Context fields
     * as object values, if desired.
     */
    static fromObject(obj) {
        return new TaskInput(InputType.OBJECT, obj);
    }
    /**
     * Use a part of the execution data or task context as task input
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution or the current task context
     * as complete payload to a task.
     */
    static fromJsonPathAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
    /**
     * Use a part of the execution data as task input
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution as complete payload
     * to a task.
     */
    static fromDataAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
    /**
     * Use a part of the task context as task input
     *
     * Use this when you want to use a subobject or string from
     * the current task context as complete payload
     * to a task.
     */
    static fromContextAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
}
exports.TaskInput = TaskInput;
/**
 * The type of task input
 */
var InputType;
(function (InputType) {
    /**
     * Use a literal string
     * This might be a JSON-encoded object, or just text.
     * valid JSON text: standalone, quote-delimited strings; objects; arrays; numbers; Boolean values; and null.
     *
     * example: `literal string`
     * example: {"json": "encoded"}
     */
    InputType[InputType["TEXT"] = 0] = "TEXT";
    /**
     * Use an object which may contain Data and Context fields
     * as object values, if desired.
     *
     * example:
     * {
     *  literal: 'literal',
     *  SomeInput: sfn.JsonPath.stringAt('$.someField')
     * }
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-data.html
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/input-output-contextobject.html
     */
    InputType[InputType["OBJECT"] = 1] = "OBJECT";
})(InputType = exports.InputType || (exports.InputType = {}));
//# sourceMappingURL=data:application/json;base64,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