"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoAttributeValue = exports.DynamoProjectionExpression = exports.DynamoReturnValues = exports.DynamoItemCollectionMetrics = exports.DynamoConsumedCapacity = void 0;
const utils_1 = require("./private/utils");
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    /**
     * The response includes the aggregate ConsumedCapacity for the operation,
     * together with ConsumedCapacity for each table and secondary index that was accessed
     */
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    /**
     * The response includes only the aggregate ConsumedCapacity for the operation.
     */
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    /**
     * No ConsumedCapacity details are included in the response.
     */
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * Determines whether item collection metrics are returned.
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    /**
     * If set to SIZE, the response includes statistics about item collections,
     * if any, that were modified during the operation.
     */
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    /**
     * If set to NONE, no statistics are returned.
     */
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    /**
     * Nothing is returned
     */
    DynamoReturnValues["NONE"] = "NONE";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * Class to generate projection expression
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * Adds the passed attribute to the chain
     *
     * @param attr Attribute name
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * Adds the array literal access for passed index
     *
     * @param index array index
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * converts and return the string expression
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
/**
 * Represents the data for an attribute.
 * Each attribute value is described as a name-value pair.
 * The name is the data type, and the value is the data itself.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_AttributeValue.html
 */
class DynamoAttributeValue {
    constructor(value) {
        this.attributeValue = value;
    }
    /**
     * Sets an attribute of type String. For example:  "S": "Hello"
     * Strings may be literal values or as JsonPath
     */
    static fromString(value) {
        return new DynamoAttributeValue({ S: value });
    }
    /**
     * Sets a literal number. For example: 1234
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumber(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Number. For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberFromString(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Binary. For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     *
     * @param value base-64 encoded string
     */
    static fromBinary(value) {
        return new DynamoAttributeValue({ B: value });
    }
    /**
     * Sets an attribute of type String Set. For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    static fromStringSet(value) {
        return new DynamoAttributeValue({ SS: value });
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumberSet(value) {
        return new DynamoAttributeValue({ NS: value.map(String) });
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberSetFromStrings(value) {
        return new DynamoAttributeValue({ NS: value });
    }
    /**
     * Sets an attribute of type Binary Set. For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    static fromBinarySet(value) {
        return new DynamoAttributeValue({ BS: value });
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    static fromMap(value) {
        return new DynamoAttributeValue({ M: utils_1.transformAttributeValueMap(value) });
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     *
     * @param value Json path that specifies state input to be used
     */
    static mapFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ 'M.$': value });
    }
    /**
     * Sets an attribute of type List. For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    static fromList(value) {
        return new DynamoAttributeValue({ L: value.map((val) => val.toObject()) });
    }
    /**
     * Sets an attribute of type Null. For example:  "NULL": true
     */
    static fromNull(value) {
        return new DynamoAttributeValue({ NULL: value });
    }
    /**
     * Sets an attribute of type Boolean. For example:  "BOOL": true
     */
    static fromBoolean(value) {
        return new DynamoAttributeValue({ BOOL: value });
    }
    /**
     * Sets an attribute of type Boolean from state input through Json path.
     * For example:  "BOOL": true
     *
     * @param value Json path that specifies state input to be used
     */
    static booleanFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ BOOL: value.toString() });
    }
    /**
     * Returns the DynamoDB attribute value
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
//# sourceMappingURL=data:application/json;base64,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