"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.arnForParameterName = exports.AUTOGEN_MARKER = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
exports.AUTOGEN_MARKER = '$$autogen$$';
/**
 * Renders an ARN for an SSM parameter given a parameter name.
 * @param scope definition scope
 * @param parameterName the parameter name to include in the ARN
 * @param physicalName optional physical name specified by the user (to auto-detect separator)
 */
function arnForParameterName(scope, parameterName, options = {}) {
    const physicalName = options.physicalName;
    const nameToValidate = physicalName || parameterName;
    if (!core_1.Token.isUnresolved(nameToValidate) && nameToValidate.includes('/') && !nameToValidate.startsWith('/')) {
        throw new Error(`Parameter names must be fully qualified (if they include "/" they must also begin with a "/"): ${nameToValidate}`);
    }
    return core_1.Stack.of(scope).formatArn({
        service: 'ssm',
        resource: 'parameter',
        sep: isSimpleName() ? '/' : '',
        resourceName: parameterName,
    });
    /**
     * Determines the ARN separator for this parameter: if we have a concrete
     * parameter name (or explicitly defined physical name), we will parse them
     * and decide whether a "/" is needed or not. Otherwise, users will have to
     * explicitly specify `simpleName` when they import the ARN.
     */
    function isSimpleName() {
        // look for a concrete name as a hint for determining the separator
        const concreteName = !core_1.Token.isUnresolved(parameterName) ? parameterName : physicalName;
        if (!concreteName || core_1.Token.isUnresolved(concreteName)) {
            if (options.simpleName === undefined) {
                throw new Error('Unable to determine ARN separator for SSM parameter since the parameter name is an unresolved token. Use "fromAttributes" and specify "simpleName" explicitly');
            }
            return options.simpleName;
        }
        const result = !concreteName.startsWith('/');
        // if users explicitly specify the separator and it conflicts with the one we need, it's an error.
        if (options.simpleName !== undefined && options.simpleName !== result) {
            if (concreteName === exports.AUTOGEN_MARKER) {
                throw new Error('If "parameterName" is not explicitly defined, "simpleName" must be "true" or undefined since auto-generated parameter names always have simple names');
            }
            throw new Error(`Parameter name "${concreteName}" is ${result ? 'a simple name' : 'not a simple name'}, but "simpleName" was explicitly set to ${options.simpleName}. Either omit it or set it to ${result}`);
        }
        return result;
    }
}
exports.arnForParameterName = arnForParameterName;
//# sourceMappingURL=data:application/json;base64,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