"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringListParameter = exports.StringParameter = exports.ParameterTier = exports.ParameterType = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ssm = require("./ssm.generated");
const util_1 = require("./util");
/**
 * Basic features shared across all types of SSM Parameters.
 */
class ParameterBase extends core_1.Resource {
    grantRead(grantee) {
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'ssm:DescribeParameters',
                'ssm:GetParameters',
                'ssm:GetParameter',
                'ssm:GetParameterHistory',
            ],
            resourceArns: [this.parameterArn],
        });
    }
    grantWrite(grantee) {
        if (this.encryptionKey) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['ssm:PutParameter'],
            resourceArns: [this.parameterArn],
        });
    }
}
/**
 * SSM parameter type
 */
var ParameterType;
(function (ParameterType) {
    /**
     * String
     */
    ParameterType["STRING"] = "String";
    /**
     * Secure String
     * Parameter Store uses an AWS Key Management Service (KMS) customer master key (CMK) to encrypt the parameter value.
     */
    ParameterType["SECURE_STRING"] = "SecureString";
    /**
     * String List
     */
    ParameterType["STRING_LIST"] = "StringList";
    /**
     * An Amazon EC2 image ID, such as ami-0ff8a91507f77f867
     */
    ParameterType["AWS_EC2_IMAGE_ID"] = "AWS::EC2::Image::Id";
})(ParameterType = exports.ParameterType || (exports.ParameterType = {}));
/**
 * SSM parameter tier
 */
var ParameterTier;
(function (ParameterTier) {
    /**
     * String
     */
    ParameterTier["ADVANCED"] = "Advanced";
    /**
     * String
     */
    ParameterTier["INTELLIGENT_TIERING"] = "Intelligent-Tiering";
    /**
     * String
     */
    ParameterTier["STANDARD"] = "Standard";
})(ParameterTier = exports.ParameterTier || (exports.ParameterTier = {}));
/**
 * Creates a new String SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringParameter extends ParameterBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.allowedPattern) {
            _assertValidValue(props.stringValue, props.allowedPattern);
        }
        if (this.physicalName.length > 2048) {
            throw new Error('Name cannot be longer than 2048 characters.');
        }
        if (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 1024) {
            throw new Error('Description cannot be longer than 1024 characters.');
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            tier: props.tier,
            type: props.type || ParameterType.STRING,
            value: props.stringValue,
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = util_1.arnForParameterName(this, this.parameterName, {
            physicalName: props.parameterName || util_1.AUTOGEN_MARKER,
            simpleName: props.simpleName,
        });
        this.parameterType = resource.attrType;
        this.stringValue = resource.attrValue;
    }
    /**
     * Imports an external string parameter by name.
     */
    static fromStringParameterName(scope, id, stringParameterName) {
        return this.fromStringParameterAttributes(scope, id, { parameterName: stringParameterName });
    }
    /**
     * Imports an external string parameter with name and optional version.
     */
    static fromStringParameterAttributes(scope, id, attrs) {
        if (!attrs.parameterName) {
            throw new Error('parameterName cannot be an empty string');
        }
        const type = attrs.type || ParameterType.STRING;
        const stringValue = attrs.version
            ? new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, `${attrs.parameterName}:${attrs.version}`).toString()
            : new core_1.CfnParameter(scope, `${id}.Parameter`, { type: `AWS::SSM::Parameter::Value<${type}>`, default: attrs.parameterName }).valueAsString;
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = util_1.arnForParameterName(this, attrs.parameterName, { simpleName: attrs.simpleName });
                this.parameterType = type;
                this.stringValue = stringValue;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a secure string parameter from the SSM parameter store.
     */
    static fromSecureStringParameterAttributes(scope, id, attrs) {
        const stringValue = new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM_SECURE, `${attrs.parameterName}:${attrs.version}`).toString();
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = util_1.arnForParameterName(this, attrs.parameterName, { simpleName: attrs.simpleName });
                this.parameterType = ParameterType.SECURE_STRING;
                this.stringValue = stringValue;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Reads the value of an SSM parameter during synthesis through an
     * environmental context provider.
     *
     * Requires that the stack this scope is defined in will have explicit
     * account/region information. Otherwise, it will fail during synthesis.
     */
    static valueFromLookup(scope, parameterName) {
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SSM_PARAMETER_PROVIDER,
            props: { parameterName },
            dummyValue: `dummy-value-for-${parameterName}`,
        }).value;
        return value;
    }
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForStringParameter(scope, parameterName, version) {
        return StringParameter.valueForTypedStringParameter(scope, parameterName, ParameterType.STRING, version);
    }
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param type The type of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForTypedStringParameter(scope, parameterName, type = ParameterType.STRING, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromStringParameterAttributes(stack, id, { parameterName, version, type }).stringValue;
    }
    /**
     * Returns a token that will resolve (during deployment)
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter
     * @param version The parameter version (required for secure strings)
     */
    static valueForSecureStringParameter(scope, parameterName, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromSecureStringParameterAttributes(stack, id, { parameterName, version }).stringValue;
    }
}
exports.StringParameter = StringParameter;
/**
 * Creates a new StringList SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringListParameter extends ParameterBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.stringListValue.find(str => !core_1.Token.isUnresolved(str) && str.indexOf(',') !== -1)) {
            throw new Error('Values of a StringList SSM Parameter cannot contain the \',\' character. Use a string parameter instead.');
        }
        if (props.allowedPattern && !core_1.Token.isUnresolved(props.stringListValue)) {
            props.stringListValue.forEach(str => _assertValidValue(str, props.allowedPattern));
        }
        if (this.physicalName.length > 2048) {
            throw new Error('Name cannot be longer than 2048 characters.');
        }
        if (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 1024) {
            throw new Error('Description cannot be longer than 1024 characters.');
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            tier: props.tier,
            type: ParameterType.STRING_LIST,
            value: props.stringListValue.join(','),
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = util_1.arnForParameterName(this, this.parameterName, {
            physicalName: props.parameterName || util_1.AUTOGEN_MARKER,
            simpleName: props.simpleName,
        });
        this.parameterType = resource.attrType;
        this.stringListValue = core_1.Fn.split(',', resource.attrValue);
    }
    /**
     * Imports an external parameter of type string list.
     * Returns a token and should not be parsed.
     */
    static fromStringListParameterName(scope, id, stringListParameterName) {
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = stringListParameterName;
                this.parameterArn = util_1.arnForParameterName(this, this.parameterName);
                this.parameterType = ParameterType.STRING_LIST;
                this.stringListValue = core_1.Fn.split(',', new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, stringListParameterName).toString());
            }
        }
        return new Import(scope, id);
    }
}
exports.StringListParameter = StringListParameter;
/**
 * Validates whether a supplied value conforms to the allowedPattern, granted neither is an unresolved token.
 *
 * @param value          the value to be validated.
 * @param allowedPattern the regular expression to use for validation.
 *
 * @throws if the ``value`` does not conform to the ``allowedPattern`` and neither is an unresolved token (per
 *         ``cdk.unresolved``).
 */
function _assertValidValue(value, allowedPattern) {
    if (core_1.Token.isUnresolved(value) || core_1.Token.isUnresolved(allowedPattern)) {
        // Unable to perform validations against unresolved tokens
        return;
    }
    if (!new RegExp(allowedPattern).test(value)) {
        throw new Error(`The supplied value (${value}) does not match the specified allowedPattern (${allowedPattern})`);
    }
}
function makeIdentityForImportedValue(parameterName) {
    return `SsmParameterValue:${parameterName}:C96584B6-F00A-464E-AD19-53AFF4B05118`;
}
//# sourceMappingURL=data:application/json;base64,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