import * as ec2 from "../../aws-ec2";
import * as kms from "../../aws-kms";
import * as secretsmanager from "../../aws-secretsmanager";
import { Duration, SecretValue } from "../../core";
import { IParameterGroup } from './parameter-group';
/**
 * Instance properties for database instances
 */
export interface InstanceProps {
    /**
     * What type of instance to start for the replicas.
     *
     * @default - t3.medium (or, more precisely, db.t3.medium)
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * What subnets to run the RDS instances in.
     *
     * Must be at least 2 subnets in two different AZs.
     */
    readonly vpc: ec2.IVpc;
    /**
     * Where to place the instances within the VPC
     *
     * @default - the Vpc default strategy if not specified.
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * Security group.
     *
     * @default a new security group is created.
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * The DB parameter group to associate with the instance.
     *
     * @default no parameter group
     */
    readonly parameterGroup?: IParameterGroup;
}
/**
 * Backup configuration for RDS databases
 *
 * @default - The retention period for automated backups is 1 day.
 * The preferred backup window will be a 30-minute window selected at random
 * from an 8-hour block of time for each AWS Region.
 * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
 */
export interface BackupProps {
    /**
     * How many days to retain the backup
     */
    readonly retention: Duration;
    /**
     * A daily time range in 24-hours UTC format in which backups preferably execute.
     *
     * Must be at least 30 minutes long.
     *
     * Example: '01:00-02:00'
     *
     * @default - a 30-minute window selected at random from an 8-hour block of
     * time for each AWS Region. To see the time blocks available, see
     * https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
     */
    readonly preferredWindow?: string;
}
/**
 * Username and password combination
 */
export interface Login {
    /**
     * Username
     */
    readonly username: string;
    /**
     * Password
     *
     * Do not put passwords in your CDK code directly.
     *
     * @default a Secrets Manager generated password
     */
    readonly password?: SecretValue;
    /**
     * KMS encryption key to encrypt the generated secret.
     *
     * @default default master key
     */
    readonly encryptionKey?: kms.IKey;
}
/**
 * Options to add the multi user rotation
 */
export interface RotationMultiUserOptions {
    /**
     * The secret to rotate. It must be a JSON string with the following format:
     * ```
     * {
     *   "engine": <required: database engine>,
     *   "host": <required: instance host name>,
     *   "username": <required: username>,
     *   "password": <required: password>,
     *   "dbname": <optional: database name>,
     *   "port": <optional: if not specified, default port will be used>,
     *   "masterarn": <required: the arn of the master secret which will be used to create users/change passwords>
     * }
     * ```
     */
    readonly secret: secretsmanager.ISecret;
    /**
     * Specifies the number of days after the previous rotation before
     * Secrets Manager triggers the next automatic rotation.
     *
     * @default Duration.days(30)
     */
    readonly automaticallyAfter?: Duration;
}
