"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceReadReplica = exports.DatabaseInstanceFromSnapshot = exports.DatabaseInstance = exports.PerformanceInsightRetention = exports.StorageType = exports.LicenseModel = exports.DatabaseInstanceBase = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a new db proxy to this instance.
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromInstance(this),
            ...options,
        });
    }
    grantConnect(grantee) {
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant connect when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [this.instanceArn],
        });
    }
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            sep: ':',
            resourceName: this.instanceIdentifier,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
exports.DatabaseInstanceBase = DatabaseInstanceBase;
/**
 * The license model.
 */
var LicenseModel;
(function (LicenseModel) {
    /**
     * License included.
     */
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    /**
     * Bring your own licencse.
     */
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    /**
     * General public license.
     */
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * The type of storage.
 */
var StorageType;
(function (StorageType) {
    /**
     * Standard.
     */
    StorageType["STANDARD"] = "standard";
    /**
     * General purpose (SSD).
     */
    StorageType["GP2"] = "gp2";
    /**
     * Provisioned IOPS (SSD).
     */
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * The retention period for Performance Insight.
 */
var PerformanceInsightRetention;
(function (PerformanceInsightRetention) {
    /**
     * Default retention period of 7 days.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["DEFAULT"] = 7] = "DEFAULT";
    /**
     * Long term retention period of 2 years.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["LONG_TERM"] = 731] = "LONG_TERM";
})(PerformanceInsightRetention = exports.PerformanceInsightRetention || (exports.PerformanceInsightRetention = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcPlacement = props.vpcPlacement;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcPlacement);
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'SubnetGroup', {
            dbSubnetGroupDescription: `Subnet group for ${this.node.id} database`,
            subnetIds,
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.numberValue({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const deletionProtection = props.deletionProtection !== undefined ? props.deletionProtection : true;
        const storageType = props.storageType || StorageType.GP2;
        const iops = storageType === StorageType.IO1 ? (props.iops || 1000) : undefined;
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.enableIamAuthentication = props.iamAuthentication;
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: props.backupRetention ? props.backupRetention.toDays() : undefined,
            copyTagsToSnapshot: props.copyTagsToSnapshot !== undefined ? props.copyTagsToSnapshot : true,
            dbInstanceClass: core_1.Lazy.stringValue({ produce: () => `db.${this.instanceType}` }),
            dbInstanceIdentifier: props.instanceIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection,
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: core_1.Lazy.anyValue({ produce: () => this.enableIamAuthentication }),
            enablePerformanceInsights: props.enablePerformanceInsights,
            iops,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            multiAz: props.multiAz,
            optionGroupName: props.optionGroup && props.optionGroup.optionGroupName,
            performanceInsightsKmsKeyId: props.enablePerformanceInsights
                ? props.performanceInsightEncryptionKey && props.performanceInsightEncryptionKey.keyArn
                : undefined,
            performanceInsightsRetentionPeriod: props.enablePerformanceInsights
                ? (props.performanceInsightRetention || PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: props.port ? props.port.toString() : undefined,
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: props.vpcPlacement && props.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC,
            storageType,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
        };
    }
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new logs.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        props.engine.bindToInstance(this, props);
        this.instanceType = (_a = props.instanceType) !== null && _a !== void 0 ? _a : ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
        const instanceParameterGroupConfig = (_b = props.parameterGroup) === null || _b === void 0 ? void 0 : _b.bindToInstance({});
        this.sourceCfnProps = {
            ...this.newCfnProps,
            allocatedStorage: props.allocatedStorage ? props.allocatedStorage.toString() : '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
            engine: props.engine.engineType,
            engineVersion: (_c = props.engine.engineVersion) === null || _c === void 0 ? void 0 : _c.fullVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
        };
    }
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this instance.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (!props.masterUserPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUsername,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstance = DatabaseInstance;
/**
 * A database instance restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (props.generateMasterUserPassword) {
            if (!props.masterUsername) { // We need the master username to include it in the generated secret
                throw new Error('`masterUsername` must be specified when `generateMasterUserPassword` is set to true.');
            }
            if (props.masterUserPassword) {
                throw new Error('Cannot specify `masterUserPassword` when `generateMasterUserPassword` is set to true.');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        else {
            if (props.masterUsername) { // It's not possible to change the master username of a RDS instance
                throw new Error('Cannot specify `masterUsername` when `generateMasterUserPassword` is set to false.');
            }
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
/**
 * A read replica database instance.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceType = props.instanceType;
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        this.setLogRetention();
    }
}
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
function applyInstanceDeletionPolicy(cfnDbInstance, removalPolicy) {
    if (!removalPolicy) {
        // the default DeletionPolicy is 'Snapshot', which is fine,
        // but we should also make it 'Snapshot' for UpdateReplace policy
        cfnDbInstance.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
    }
    else {
        // just apply whatever removal policy the customer explicitly provided
        cfnDbInstance.applyRemovalPolicy(removalPolicy);
    }
}
//# sourceMappingURL=data:application/json;base64,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