"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Runtime = exports.RuntimeFamily = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
var RuntimeFamily;
(function (RuntimeFamily) {
    RuntimeFamily[RuntimeFamily["NODEJS"] = 0] = "NODEJS";
    RuntimeFamily[RuntimeFamily["JAVA"] = 1] = "JAVA";
    RuntimeFamily[RuntimeFamily["PYTHON"] = 2] = "PYTHON";
    RuntimeFamily[RuntimeFamily["DOTNET_CORE"] = 3] = "DOTNET_CORE";
    RuntimeFamily[RuntimeFamily["GO"] = 4] = "GO";
    RuntimeFamily[RuntimeFamily["RUBY"] = 5] = "RUBY";
    RuntimeFamily[RuntimeFamily["OTHER"] = 6] = "OTHER";
})(RuntimeFamily = exports.RuntimeFamily || (exports.RuntimeFamily = {}));
/**
 * Lambda function runtime environment.
 *
 * If you need to use a runtime name that doesn't exist as a static member, you
 * can instantiate a `Runtime` object, e.g: `new Runtime('nodejs99.99')`.
 */
class Runtime {
    constructor(name, family, props = {}) {
        var _a;
        this.name = name;
        this.supportsInlineCode = !!props.supportsInlineCode;
        this.family = family;
        const imageName = (_a = props.bundlingDockerImage) !== null && _a !== void 0 ? _a : `amazon/aws-sam-cli-build-image-${name}`;
        this.bundlingDockerImage = core_1.BundlingDockerImage.fromRegistry(imageName);
        Runtime.ALL.push(this);
    }
    toString() {
        return this.name;
    }
    runtimeEquals(other) {
        return other.name === this.name &&
            other.family === this.family &&
            other.supportsInlineCode === this.supportsInlineCode;
    }
}
exports.Runtime = Runtime;
/** A list of all known `Runtime`'s. */
Runtime.ALL = new Array();
/**
 * The NodeJS runtime (nodejs)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS = new Runtime('nodejs', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 4.3 runtime (nodejs4.3)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_4_3 = new Runtime('nodejs4.3', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 6.10 runtime (nodejs6.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_6_10 = new Runtime('nodejs6.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 8.10 runtime (nodejs8.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_8_10 = new Runtime('nodejs8.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 10.x runtime (nodejs10.x)
 */
Runtime.NODEJS_10_X = new Runtime('nodejs10.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 12.x runtime (nodejs12.x)
 */
Runtime.NODEJS_12_X = new Runtime('nodejs12.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The Python 2.7 runtime (python2.7)
 */
Runtime.PYTHON_2_7 = new Runtime('python2.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.6 runtime (python3.6)
 */
Runtime.PYTHON_3_6 = new Runtime('python3.6', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.7 runtime (python3.7)
 */
Runtime.PYTHON_3_7 = new Runtime('python3.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.8 runtime (python3.8)
 */
Runtime.PYTHON_3_8 = new Runtime('python3.8', RuntimeFamily.PYTHON);
/**
 * The Java 8 runtime (java8)
 */
Runtime.JAVA_8 = new Runtime('java8', RuntimeFamily.JAVA);
/**
 * The Java 11 runtime (java11)
 */
Runtime.JAVA_11 = new Runtime('java11', RuntimeFamily.JAVA);
/**
 * The .NET Core 1.0 runtime (dotnetcore1.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_1 = new Runtime('dotnetcore1.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.0 runtime (dotnetcore2.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_2 = new Runtime('dotnetcore2.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.1 runtime (dotnetcore2.1)
 */
Runtime.DOTNET_CORE_2_1 = new Runtime('dotnetcore2.1', RuntimeFamily.DOTNET_CORE, {
    bundlingDockerImage: 'lambci/lambda:build-dotnetcore2.1',
});
/**
 * The .NET Core 3.1 runtime (dotnetcore3.1)
 */
Runtime.DOTNET_CORE_3_1 = new Runtime('dotnetcore3.1', RuntimeFamily.DOTNET_CORE, {
    bundlingDockerImage: 'lambci/lambda:build-dotnetcore3.1',
});
/**
 * The Go 1.x runtime (go1.x)
 */
Runtime.GO_1_X = new Runtime('go1.x', RuntimeFamily.GO, {
    bundlingDockerImage: 'lambci/lambda:build-go1.x',
});
/**
 * The Ruby 2.5 runtime (ruby2.5)
 */
Runtime.RUBY_2_5 = new Runtime('ruby2.5', RuntimeFamily.RUBY);
/**
 * The Ruby 2.7 runtime (ruby2.7)
 */
Runtime.RUBY_2_7 = new Runtime('ruby2.7', RuntimeFamily.RUBY);
/**
 * The custom provided runtime (provided)
 */
Runtime.PROVIDED = new Runtime('provided', RuntimeFamily.OTHER);
//# sourceMappingURL=data:application/json;base64,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