"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.QualifiedFunctionBase = exports.FunctionBase = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const event_invoke_config_1 = require("./event-invoke-config");
const event_source_mapping_1 = require("./event-source-mapping");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
class FunctionBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Mapping of invocation principals to grants. Used to de-dupe `grantInvoke()` calls.
         * @internal
         */
        this._invocationGrants = {};
    }
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     * @param permission The permission to grant to this Lambda function. @see Permission for details.
     */
    addPermission(id, permission) {
        if (!this.canCreatePermissions) {
            // FIXME: Report metadata
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const action = permission.action || 'lambda:InvokeFunction';
        const scope = permission.scope || this;
        new lambda_generated_1.CfnPermission(scope, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: permission.sourceAccount,
            sourceArn: permission.sourceArn,
        });
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    get latestVersion() {
        if (!this._latestVersion) {
            this._latestVersion = new LatestVersion(this);
        }
        return this._latestVersion;
    }
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    addEventSourceMapping(id, options) {
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options,
        });
    }
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee) {
        const identifier = `Invoke${grantee.grantPrincipal}`; // calls the .toString() of the principal
        // Memoize the result so subsequent grantInvoke() calls are idempotent
        let grant = this._invocationGrants[identifier];
        if (!grant) {
            grant = iam.Grant.addToPrincipalOrResource({
                grantee,
                actions: ['lambda:InvokeFunction'],
                resourceArns: [this.functionArn],
                // Fake resource-like object on which to call addToResourcePolicy(), which actually
                // calls addPermission()
                resource: {
                    addToResourcePolicy: (_statement) => {
                        // Couldn't add permissions to the principal, so add them locally.
                        this.addPermission(identifier, {
                            principal: grantee.grantPrincipal,
                            action: 'lambda:InvokeFunction',
                        });
                        return { statementAdded: true, policyDependable: this._functionNode().findChild(identifier) };
                    },
                    node: this.node,
                    stack: this.stack,
                    env: this.env,
                },
            });
            this._invocationGrants[identifier] = grant;
        }
        return grant;
    }
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source) {
        source.bind(this);
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this,
            ...options,
        });
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * For use internally for constructs, when the tree is set up in non-standard ways. Ex: SingletonFunction.
     * @internal
     */
    _functionNode() {
        return this.node;
    }
    parsePermissionPrincipal(principal) {
        if (!principal) {
            return undefined;
        }
        // use duck-typing, not instance of
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if ('service' in principal) {
            return principal.service;
        }
        if ('arn' in principal) {
            return principal.arn;
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ArnPrincipal, ServicePrincipal');
    }
}
exports.FunctionBase = FunctionBase;
class QualifiedFunctionBase extends FunctionBase {
    constructor() {
        super(...arguments);
        this.permissionsNode = this.node;
    }
    get latestVersion() {
        return this.lambda.latestVersion;
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the qualified function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this.lambda,
            qualifier: this.qualifier,
            ...options,
        });
    }
}
exports.QualifiedFunctionBase = QualifiedFunctionBase;
/**
 * The $LATEST version of a function, useful when attempting to create aliases.
 */
class LatestVersion extends FunctionBase {
    constructor(lambda) {
        super(lambda, '$LATEST');
        this.version = '$LATEST';
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.lambda = lambda;
    }
    get functionArn() {
        return `${this.lambda.functionArn}:${this.version}`;
    }
    get functionName() {
        return `${this.lambda.functionName}:${this.version}`;
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get latestVersion() {
        return this;
    }
    get role() {
        return this.lambda.role;
    }
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
    get edgeArn() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
}
//# sourceMappingURL=data:application/json;base64,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