"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParametersCode = exports.AssetCode = exports.InlineCode = exports.S3Code = exports.Code = void 0;
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * @deprecated use `fromBucket`
     */
    static bucket(bucket, key, objectVersion) {
        return this.fromBucket(bucket, key, objectVersion);
    }
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * @deprecated use `fromInline`
     */
    static inline(code) {
        return this.fromInline(code);
    }
    /**
     * Loads the function code from a local disk path.
     *
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static fromAsset(path, options) {
        return new AssetCode(path, options);
    }
    /**
     * @deprecated use `fromAsset`
     */
    static asset(path) {
        return this.fromAsset(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static fromCfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * @deprecated use `fromCfnParameters`
     */
    static cfnParameters(props) {
        return this.fromCfnParameters(props);
    }
    /**
     * Called after the CFN function resource has been created to allow the code
     * class to bind to it. Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindToResource(_resource, _options) {
        return;
    }
}
exports.Code = Code;
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        this.isInline = true;
        if (code.length === 0) {
            throw new Error('Lambda inline code cannot be empty');
        }
        if (code.length > 4096) {
            throw new Error('Lambda source is too large, must be <= 4096 but is ' + code.length);
        }
    }
    bind(_scope) {
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        this.isInline = false;
    }
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.path,
                ...this.options,
            });
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    bindToResource(resource, options = {}) {
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const resourceProperty = options.resourceProperty || 'Code';
        // https://github.com/aws/aws-cdk/issues/1432
        this.asset.addResourceMetadata(resource, resourceProperty);
    }
}
exports.AssetCode = AssetCode;
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
class CfnParametersCode extends Code {
    constructor(props = {}) {
        super();
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    bind(scope) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(scope, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(scope, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
        return {
            s3Location: {
                bucketName: this._bucketNameParam.valueAsString,
                objectKey: this._objectKeyParam.valueAsString,
            },
        };
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
//# sourceMappingURL=data:application/json;base64,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