"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisEventSource = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const stream_1 = require("./stream");
/**
 * Use an Amazon Kinesis stream as an event source for AWS Lambda.
 */
class KinesisEventSource extends stream_1.StreamEventSource {
    constructor(stream, props) {
        super(props);
        this.stream = stream;
        this._eventSourceMappingId = undefined;
        this.props.batchSize !== undefined && cdk.withResolved(this.props.batchSize, batchSize => {
            if (batchSize < 1 || batchSize > 10000) {
                throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize})`);
            }
        });
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`KinesisEventSource:${this.stream.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.stream.streamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.stream.grantRead(target);
        // The `grantRead` API provides all the permissions recommended by the Kinesis team for reading a stream.
        // `DescribeStream` permissions are not required to read a stream as it's covered by the `DescribeStreamSummary`
        // and `SubscribeToShard` APIs.
        // The Lambda::EventSourceMapping resource validates against the `DescribeStream` permission. So we add it explicitly.
        // FIXME This permission can be removed when the event source mapping resource drops it from validation.
        this.stream.grant(target, 'kinesis:DescribeStream');
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KinesisEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.KinesisEventSource = KinesisEventSource;
//# sourceMappingURL=data:application/json;base64,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