"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositePrincipal = exports.Anyone = exports.AnyPrincipal = exports.AccountRootPrincipal = exports.OpenIdConnectPrincipal = exports.WebIdentityPrincipal = exports.FederatedPrincipal = exports.CanonicalUserPrincipal = exports.OrganizationPrincipal = exports.ServicePrincipal = exports.AccountPrincipal = exports.ArnPrincipal = exports.PrincipalPolicyFragment = exports.PrincipalWithConditions = exports.PrincipalBase = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const region_info_1 = require("../../region-info"); // Automatically re-written from '@aws-cdk/region-info'
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        this.principalAccount = undefined;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return { statementAdded: false };
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    /**
     * Returns a new PrincipalWithConditions using this principal as the base, with the
     * passed conditions added.
     *
     * When there is a value for the same operator and key in both the principal and the
     * conditions parameter, the value from the conditions parameter will be used.
     *
     * @returns a new PrincipalWithConditions object.
     */
    withConditions(conditions) {
        return new PrincipalWithConditions(this, conditions);
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * An IAM principal with additional conditions specifying when the policy is in effect.
 *
 * For more information about conditions, see:
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html
 */
class PrincipalWithConditions {
    constructor(principal, conditions) {
        this.principal = principal;
        this.grantPrincipal = this;
        this.assumeRoleAction = this.principal.assumeRoleAction;
        this.additionalConditions = conditions;
    }
    /**
     * Add a condition to the principal
     */
    addCondition(key, value) {
        const existingValue = this.conditions[key];
        this.conditions[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Adds multiple conditions to the principal
     *
     * Values from the conditions parameter will overwrite existing values with the same operator
     * and key.
     */
    addConditions(conditions) {
        Object.entries(conditions).forEach(([key, value]) => {
            this.addCondition(key, value);
        });
    }
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    get conditions() {
        return this.mergeConditions(this.principal.policyFragment.conditions, this.additionalConditions);
    }
    get policyFragment() {
        return new PrincipalPolicyFragment(this.principal.policyFragment.principalJson, this.conditions);
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(statement) {
        return this.principal.addToPrincipalPolicy(statement);
    }
    toString() {
        return this.principal.toString();
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    mergeConditions(principalConditions, additionalConditions) {
        const mergedConditions = {};
        Object.entries(principalConditions).forEach(([operator, condition]) => {
            mergedConditions[operator] = condition;
        });
        Object.entries(additionalConditions).forEach(([operator, condition]) => {
            // merge the conditions if one of the additional conditions uses an
            // operator that's already used by the principal's conditions merge the
            // inner structure.
            const existing = mergedConditions[operator];
            if (!existing) {
                mergedConditions[operator] = condition;
                return; // continue
            }
            // if either the existing condition or the new one contain unresolved
            // tokens, fail the merge. this is as far as we go at this point.
            if (cdk.Token.isUnresolved(condition) || cdk.Token.isUnresolved(existing)) {
                throw new Error(`multiple "${operator}" conditions cannot be merged if one of them contains an unresolved token`);
            }
            mergedConditions[operator] = { ...existing, ...condition };
        });
        return mergedConditions;
    }
}
exports.PrincipalWithConditions = PrincipalWithConditions;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    /**
     *
     * @param principalJson JSON of the "Principal" section in a policy statement
     * @param conditions conditions that need to be applied to this policy
     */
    constructor(principalJson, 
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
/**
 * Specify a principal by the Amazon Resource Name (ARN).
 * You can specify AWS accounts, IAM users, Federated SAML users, IAM roles, and specific assumed-role sessions.
 * You cannot specify IAM groups or instance profiles as principals
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html
 */
class ArnPrincipal extends PrincipalBase {
    /**
     *
     * @param arn Amazon Resource Name (ARN) of the principal entity (i.e. arn:aws:iam::123456789012:user/user-name)
     */
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
/**
 * Specify AWS account ID as the principal entity in a policy to delegate authority to the account.
 */
class AccountPrincipal extends ArnPrincipal {
    /**
     *
     * @param accountId AWS account ID (i.e. 123456789012)
     */
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    /**
     *
     * @param service AWS service (i.e. sqs.amazonaws.com)
     */
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString(),
            ],
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    /**
     *
     * @param organizationId The unique identifier (ID) of an organization (i.e. o-12345abcde)
     */
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy principal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    /**
     *
     * @param canonicalUserId unique identifier assigned by AWS for every account.
     *   root user and IAM users for an account all see the same ID.
     *   (i.e. 79a59df900b949e55d96a1e698fbacedfd6e09d98eacf8f8d5218e7cd47ef2be)
     */
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
/**
 * Principal entity that represents a federated identity provider such as Amazon Cognito,
 * that can be used to provide temporary security credentials to users who have been authenticated.
 * Additional condition keys are available when the temporary security credentials are used to make a request.
 * You can use these keys to write policies that limit the access of federated users.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#condition-keys-wif
 */
class FederatedPrincipal extends PrincipalBase {
    /**
     *
     * @param federated federated identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
/**
 * A principal that represents a federated identity provider as Web Identity such as Cognito, Amazon,
 * Facebook, Google, etc.
 */
class WebIdentityPrincipal extends FederatedPrincipal {
    /**
     *
     * @param identityProvider identity provider (i.e. 'cognito-identity.amazonaws.com' for users authenticated through Cognito)
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(identityProvider, conditions = {}) {
        super(identityProvider, conditions !== null && conditions !== void 0 ? conditions : {}, 'sts:AssumeRoleWithWebIdentity');
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `WebIdentityPrincipal(${this.federated})`;
    }
}
exports.WebIdentityPrincipal = WebIdentityPrincipal;
/**
 * A principal that represents a federated identity provider as from a OpenID Connect provider.
 */
class OpenIdConnectPrincipal extends WebIdentityPrincipal {
    /**
     *
     * @param openIdConnectProvider OpenID Connect provider
     * @param conditions The conditions under which the policy is in effect.
     *   See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    constructor(openIdConnectProvider, conditions = {}) {
        super(openIdConnectProvider.openIdConnectProviderArn, conditions !== null && conditions !== void 0 ? conditions : {});
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `OpenIdConnectPrincipal(${this.federated})`;
    }
}
exports.OpenIdConnectPrincipal = OpenIdConnectPrincipal;
/**
 * Use the AWS account into which a stack is deployed as the principal entity in a policy
 */
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    toString() {
        return 'AccountRootPrincipal()';
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return 'AnyPrincipal()';
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
/**
 * Represents a principal that has multiple types of principals. A composite principal cannot
 * have conditions. i.e. multiple ServicePrincipals that form a composite principal
 */
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    /**
     * Adds IAM principals to the composite principal. Composite principals cannot have
     * conditions.
     *
     * @param principals IAM principals that will be added to the composite principal
     */
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error('Cannot add multiple principals with different "assumeRoleAction". ' +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error('Components of a CompositePrincipal must not have conditions. ' +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        return this.fn(cdk.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return '<unresolved-token>';
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || cdk.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, cdk.Aws.URL_SUFFIX);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service,
        });
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,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