"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Table = exports.TableEncryption = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const glue_generated_1 = require("./glue.generated");
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    TableEncryption["KMS"] = "SSE-KMS";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * A Glue table.
 */
class Table extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = (_a = props.s3Prefix) !== null && _a !== void 0 ? _a : '';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = props.compressed === undefined ? false : props.compressed;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new glue_generated_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    has_encrypted_data: this.encryption !== TableEncryption.UNENCRYPTED,
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: props.storedAsSubDirectories === undefined ? false : props.storedAsSubDirectories,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className,
                    },
                },
                tableType: 'EXTERNAL_TABLE',
            },
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`,
        });
        this.node.defaultChild = tableResource;
    }
    static fromTableArn(scope, id, tableArn) {
        const tableName = core_1.Fn.select(1, core_1.Fn.split('/', core_1.Stack.of(scope).parseArn(tableArn).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName,
        });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.s3Prefix);
        return ret;
    }
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.s3Prefix);
        return ret;
    }
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.s3Prefix);
        return ret;
    }
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
}
exports.Table = Table;
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but \'${column.name}\' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey,
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey,
    };
}
const readPermissions = [
    'glue:BatchDeletePartition',
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersions',
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition',
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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