"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventField = exports.RuleTargetInput = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The input to send to the event target
 */
class RuleTargetInput {
    /**
     * Pass text to the event target
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromText(text) {
        return new FieldAwareEventInput(text, InputType.Text);
    }
    /**
     * Pass text to the event target, splitting on newlines.
     *
     * This is only useful when passing to a target that does not
     * take a single argument.
     *
     * May contain strings returned by EventField.from() to substitute in parts
     * of the matched event.
     */
    static fromMultilineText(text) {
        return new FieldAwareEventInput(text, InputType.Multiline);
    }
    /**
     * Pass a JSON object to the event target
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromObject(obj) {
        return new FieldAwareEventInput(obj, InputType.Object);
    }
    /**
     * Take the event target input from a path in the event JSON
     */
    static fromEventPath(path) {
        return new LiteralEventInput({ inputPath: path });
    }
    constructor() {
    }
}
exports.RuleTargetInput = RuleTargetInput;
/**
 * Event Input that is directly derived from the construct
 */
class LiteralEventInput extends RuleTargetInput {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Return the input properties for this input object
     */
    bind(_rule) {
        return this.props;
    }
}
/**
 * Input object that can contain field replacements
 *
 * Evaluation is done in the bind() method because token resolution
 * requires access to the construct tree.
 *
 * Multiple tokens that use the same path will use the same substitution
 * key.
 *
 * One weird exception: if we're in object context, we MUST skip the quotes
 * around the placeholder. I assume this is so once a trivial string replace is
 * done later on by EventBridge, numbers are still numbers.
 *
 * So in string context:
 *
 *    "this is a string with a <field>"
 *
 * But in object context:
 *
 *    "{ \"this is the\": <field> }"
 *
 * To achieve the latter, we postprocess the JSON string to remove the surrounding
 * quotes by using a string replace.
 */
class FieldAwareEventInput extends RuleTargetInput {
    constructor(input, inputType) {
        super();
        this.input = input;
        this.inputType = inputType;
    }
    bind(rule) {
        let fieldCounter = 0;
        const pathToKey = new Map();
        const inputPathsMap = {};
        function keyForField(f) {
            const existing = pathToKey.get(f.path);
            if (existing !== undefined) {
                return existing;
            }
            fieldCounter += 1;
            const key = f.displayHint || `f${fieldCounter}`;
            pathToKey.set(f.path, key);
            return key;
        }
        const self = this;
        class EventFieldReplacer extends core_1.DefaultTokenResolver {
            constructor() {
                super(new core_1.StringConcat());
            }
            resolveToken(t, _context) {
                if (!isEventField(t)) {
                    return core_1.Token.asString(t);
                }
                const key = keyForField(t);
                if (inputPathsMap[key] && inputPathsMap[key] !== t.path) {
                    throw new Error(`Single key '${key}' is used for two different JSON paths: '${t.path}' and '${inputPathsMap[key]}'`);
                }
                inputPathsMap[key] = t.path;
                return self.keyPlaceholder(key);
            }
        }
        const stack = core_1.Stack.of(rule);
        let resolved;
        if (this.inputType === InputType.Multiline) {
            // JSONify individual lines
            resolved = core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            });
            resolved = resolved.split('\n').map(stack.toJsonString).join('\n');
        }
        else {
            resolved = stack.toJsonString(core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            }));
        }
        if (Object.keys(inputPathsMap).length === 0) {
            // Nothing special, just return 'input'
            return { input: resolved };
        }
        return {
            inputTemplate: this.unquoteKeyPlaceholders(resolved),
            inputPathsMap,
        };
    }
    /**
     * Return a template placeholder for the given key
     *
     * In object scope we'll need to get rid of surrounding quotes later on, so
     * return a bracing that's unlikely to occur naturally (like tokens).
     */
    keyPlaceholder(key) {
        if (this.inputType !== InputType.Object) {
            return `<${key}>`;
        }
        return UNLIKELY_OPENING_STRING + key + UNLIKELY_CLOSING_STRING;
    }
    /**
     * Removing surrounding quotes from any object placeholders
     *
     * Those have been put there by JSON.stringify(), but we need to
     * remove them.
     */
    unquoteKeyPlaceholders(sub) {
        if (this.inputType !== InputType.Object) {
            return sub;
        }
        return core_1.Lazy.stringValue({ produce: (ctx) => core_1.Token.asString(deepUnquote(ctx.resolve(sub))) });
        function deepUnquote(resolved) {
            if (Array.isArray(resolved)) {
                return resolved.map(deepUnquote);
            }
            else if (typeof (resolved) === 'object' && resolved !== null) {
                for (const [key, value] of Object.entries(resolved)) {
                    resolved[key] = deepUnquote(value);
                }
                return resolved;
            }
            else if (typeof (resolved) === 'string') {
                return resolved.replace(OPENING_STRING_REGEX, '<').replace(CLOSING_STRING_REGEX, '>');
            }
            return resolved;
        }
    }
}
const UNLIKELY_OPENING_STRING = '<<${';
const UNLIKELY_CLOSING_STRING = '}>>';
const OPENING_STRING_REGEX = new RegExp(regexQuote('"' + UNLIKELY_OPENING_STRING), 'g');
const CLOSING_STRING_REGEX = new RegExp(regexQuote(UNLIKELY_CLOSING_STRING + '"'), 'g');
/**
 * Represents a field in the event pattern
 */
class EventField {
    /**
     *
     * @param path the path to a field in the event pattern
     */
    constructor(path) {
        this.path = path;
        this.displayHint = this.path.replace(/^[^a-zA-Z0-9_-]+/, '').replace(/[^a-zA-Z0-9_-]/g, '-');
        Object.defineProperty(this, EVENT_FIELD_SYMBOL, { value: true });
        this.creationStack = core_1.captureStackTrace();
    }
    /**
     * Extract the event ID from the event
     */
    static get eventId() {
        return this.fromPath('$.id');
    }
    /**
     * Extract the detail type from the event
     */
    static get detailType() {
        return this.fromPath('$.detail-type');
    }
    /**
     * Extract the source from the event
     */
    static get source() {
        return this.fromPath('$.source');
    }
    /**
     * Extract the account from the event
     */
    static get account() {
        return this.fromPath('$.account');
    }
    /**
     * Extract the time from the event
     */
    static get time() {
        return this.fromPath('$.time');
    }
    /**
     * Extract the region from the event
     */
    static get region() {
        return this.fromPath('$.region');
    }
    /**
     * Extract a custom JSON path from the event
     */
    static fromPath(path) {
        return new EventField(path).toString();
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    /**
     * Convert the path to the field in the event pattern to JSON
     */
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.EventField = EventField;
var InputType;
(function (InputType) {
    InputType[InputType["Object"] = 0] = "Object";
    InputType[InputType["Text"] = 1] = "Text";
    InputType[InputType["Multiline"] = 2] = "Multiline";
})(InputType || (InputType = {}));
function isEventField(x) {
    return EVENT_FIELD_SYMBOL in x;
}
const EVENT_FIELD_SYMBOL = Symbol.for('@aws-cdk/aws-events.EventField');
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, '\\$&');
}
//# sourceMappingURL=data:application/json;base64,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