"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GelfLogDriver = exports.GelfCompressionType = void 0;
const log_driver_1 = require("./log-driver");
const utils_1 = require("./utils");
/**
 * The type of compression the GELF driver uses to compress each log message.
 */
var GelfCompressionType;
(function (GelfCompressionType) {
    GelfCompressionType["GZIP"] = "gzip";
    GelfCompressionType["ZLIB"] = "zlib";
    GelfCompressionType["NONE"] = "none";
})(GelfCompressionType = exports.GelfCompressionType || (exports.GelfCompressionType = {}));
/**
 * A log driver that sends log information to journald Logs.
 */
class GelfLogDriver extends log_driver_1.LogDriver {
    /**
     * Constructs a new instance of the GelfLogDriver class.
     *
     * @param props the gelf log driver configuration options.
     */
    constructor(props) {
        super();
        this.props = props;
        // Validation
        if (props.compressionLevel) {
            utils_1.ensureInRange(props.compressionLevel, -1, 9);
        }
        if (props.tcpMaxReconnect) {
            utils_1.ensurePositiveInteger(props.tcpMaxReconnect);
        }
    }
    /**
     * Called when the log driver is configured on a container
     */
    bind(_scope, _containerDefinition) {
        return {
            logDriver: 'gelf',
            options: utils_1.stringifyOptions({
                'gelf-address': this.props.address,
                'gelf-compression-type': this.props.compressionType,
                'gelf-compression-level': this.props.compressionLevel,
                'gelf-tcp-max-reconnect': this.props.tcpMaxReconnect,
                'gelf-tcp-reconnect-delay': this.props.tcpReconnectDelay && this.props.tcpReconnectDelay.toSeconds(),
                ...utils_1.renderCommonLogDriverOptions(this.props),
            }),
        };
    }
}
exports.GelfLogDriver = GelfLogDriver;
//# sourceMappingURL=data:application/json;base64,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