"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroup = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * Return whether the indicated object is a security group
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    get uniqueId() {
        return this.node.uniqueId;
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toIngressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toEgressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
/**
 * Determine where to parent a new ingress/egress rule
 *
 * A SecurityGroup rule is parented under the group it's related to, UNLESS
 * we're in a cross-stack scenario with another Security Group. In that case,
 * we respect the 'remoteRule' flag and will parent under the other security
 * group.
 *
 * This is necessary to avoid cyclic dependencies between stacks, since both
 * ingress and egress rules will reference both security groups, and a naive
 * parenting will lead to the following situation:
 *
 *   ╔════════════════════╗         ╔════════════════════╗
 *   ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
 *   ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
 *   ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
 *   ║        ▲           ║ │   │   ║          ▲         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
 *   ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
 *   ║  └───────────┘     ║         ║    └───────────┘   ║
 *   ║                    ║         ║                    ║
 *   ╚════════════════════╝         ╚════════════════════╝
 *
 * By having the ability to switch the parent, we avoid the cyclic reference by
 * keeping all rules in a single stack.
 *
 * If this happens, we also have to change the construct ID, because
 * otherwise we might have two objects with the same ID if we have
 * multiple reversed security group relationships.
 *
 *   ╔═══════════════════════════════════╗
 *   ║┌───────────┐                      ║
 *   ║│  GroupB   │                      ║
 *   ║└───────────┘                      ║
 *   ║      ▲                            ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├────"from A"──│ IngressB  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├─────"to B"───│  EgressA  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      └─────"to B"───│  EgressC  │ ║  <-- oops
 *   ║                     └───────────┘ ║
 *   ╚═══════════════════════════════════╝
 */
function determineRuleScope(group, peer, connection, fromTo, remoteRule) {
    if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(group, peer)) {
        // Reversed
        const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
        return [peer, `${group.uniqueId}:${connection} ${reversedFromTo}`];
    }
    else {
        // Regular (do old ID escaping to in order to not disturb existing deployments)
        return [group, `${fromTo} ${renderPeer(peer)}:${connection}`.replace('/', '_')];
    }
}
function renderPeer(peer) {
    return core_1.Token.isUnresolved(peer.uniqueId) ? '{IndirectPeer}' : peer.uniqueId;
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * Creates an Amazon EC2 security group within a VPC.
 *
 * Security Groups act like a firewall with a set of rules, and are associated
 * with any AWS resource that has or creates Elastic Network Interfaces (ENIs).
 * A typical example of a resource that has a security group is an Instance (or
 * Auto Scaling Group of instances)
 *
 * If you are defining new infrastructure in CDK, there is a good chance you
 * won't have to interact with this class at all. Like IAM Roles, Security
 * Groups need to exist to control access between AWS resources, but CDK will
 * automatically generate and populate them with least-privilege permissions
 * for you so you can concentrate on your business logic.
 *
 * All Constructs that require Security Groups will create one for you if you
 * don't specify one at construction. After construction, you can selectively
 * allow connections to and between constructs via--for example-- the `instance.connections`
 * object. Think of it as "allowing connections to your instance", rather than
 * "adding ingress rules a security group". See the [Allowing
 * Connections](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#allowing-connections)
 * section in the library documentation for examples.
 *
 * Direct manipulation of the Security Group through `addIngressRule` and
 * `addEgressRule` is possible, but mutation through the `.connections` object
 * is recommended. If you peer two constructs with security groups this way,
 * appropriate rules will be created in both.
 *
 * If you have an existing security group you want to use in your CDK application,
 * you would import it like this:
 *
 * ```ts
 * const securityGroup = SecurityGroup.fromSecurityGroupId(this, 'SG', 'sg-12345', {
 *   mutable: false
 * });
 * ```
 */
class SecurityGroup extends SecurityGroupBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName,
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.anyValue({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.anyValue({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * Import an existing security group into this app.
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        class MutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule({
            ...peer.toIngressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        });
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            this.node.addWarning('Ignoring Egress rule since \'allowAllOutbound\' is set to true; To add customize rules, set allowAllOutbound=false on the SecurityGroup');
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = {
            ...peer.toEgressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        };
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.allowAllOutbound) {
            this.directEgressRules.push(ALLOW_ALL_RULE);
        }
        else {
            this.directEgressRules.push(MATCH_NO_TRAFFIC);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
        if (i > -1) {
            this.directEgressRules.splice(i, 1);
        }
    }
}
exports.SecurityGroup = SecurityGroup;
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86,
};
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,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