"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AclTraffic = exports.AclCidr = void 0;
/**
 * Either an IPv4 or an IPv6 CIDR
 *
 * @experimental
 */
class AclCidr {
    /**
     * An IP network range in CIDR notation (for example, 172.16.0.0/24).
     */
    static ipv4(ipv4Cidr) {
        return new AclCidrImpl({
            cidrBlock: ipv4Cidr,
        });
    }
    /**
     * The CIDR containing all IPv4 addresses (i.e., 0.0.0.0/0)
     */
    static anyIpv4() {
        return AclCidr.ipv4('0.0.0.0/0');
    }
    /**
     * An IPv6 network range in CIDR notation (for example, 2001:db8::/48)
     */
    static ipv6(ipv6Cidr) {
        return new AclCidrImpl({
            ipv6CidrBlock: ipv6Cidr,
        });
    }
    /**
     * The CIDR containing all IPv6 addresses (i.e., ::/0)
     */
    static anyIpv6() {
        return AclCidr.ipv6('::/0');
    }
}
exports.AclCidr = AclCidr;
class AclCidrImpl extends AclCidr {
    constructor(config) {
        super();
        this.config = config;
    }
    toCidrConfig() {
        return this.config;
    }
}
/**
 * The traffic that is configured using a Network ACL entry
 *
 * @experimental
 */
class AclTraffic {
    /**
     * Apply the ACL entry to all traffic
     */
    static allTraffic() {
        return new AclTrafficImpl({
            protocol: -1,
        });
    }
    /**
     * Apply the ACL entry to ICMP traffic of given type and code
     */
    static icmp(props) {
        return new AclTrafficImpl({
            protocol: 1,
            icmp: props,
        });
    }
    /**
     * Apply the ACL entry to ICMPv6 traffic of given type and code
     *
     * Requires an IPv6 CIDR block.
     */
    static icmpv6(props) {
        return new AclTrafficImpl({
            protocol: 58,
            icmp: props,
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port
     */
    static tcpPort(port) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: port,
                to: port,
            },
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port range
     */
    static tcpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: startPort,
                to: endPort,
            },
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port
     */
    static udpPort(port) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: port,
                to: port,
            },
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port range
     */
    static udpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: startPort,
                to: endPort,
            },
        });
    }
}
exports.AclTraffic = AclTraffic;
class AclTrafficImpl extends AclTraffic {
    constructor(config) {
        super();
        this.config = config;
    }
    toTrafficConfig() {
        return this.config;
    }
}
//# sourceMappingURL=data:application/json;base64,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