"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildAction = exports.CodeBuildActionType = void 0;
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const __1 = require("..");
const action_1 = require("../action");
/**
 * The type of the CodeBuild action that determines its CodePipeline Category -
 * Build, or Test.
 * The default is Build.
 */
var CodeBuildActionType;
(function (CodeBuildActionType) {
    /**
     * The action will have the Build Category.
     * This is the default.
     */
    CodeBuildActionType[CodeBuildActionType["BUILD"] = 0] = "BUILD";
    /**
     * The action will have the Test Category.
     */
    CodeBuildActionType[CodeBuildActionType["TEST"] = 1] = "TEST";
})(CodeBuildActionType = exports.CodeBuildActionType || (exports.CodeBuildActionType = {}));
/**
 * CodePipeline build action that uses AWS CodeBuild.
 */
class CodeBuildAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: props.type === CodeBuildActionType.TEST
                ? codepipeline.ActionCategory.TEST
                : codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.input, ...props.extraInputs || []],
            resource: props.project,
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the CodeBuild project this action points to.
     * Variables in CodeBuild actions are defined using the 'exported-variables' subsection of the 'env'
     * section of the buildspec.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'exported-variables' section of the buildspec
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-syntax
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        // check for a cross-account action if there are any outputs
        if ((this.actionProperties.outputs || []).length > 0) {
            const pipelineStack = cdk.Stack.of(scope);
            const projectStack = cdk.Stack.of(this.props.project);
            if (pipelineStack.account !== projectStack.account) {
                throw new Error('A cross-account CodeBuild action cannot have outputs. ' +
                    'This is a known CodeBuild limitation. ' +
                    'See https://github.com/aws/aws-cdk/issues/4169 for details');
            }
        }
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.project.projectArn],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (this.props.project.role) {
            if ((this.actionProperties.outputs || []).length > 0) {
                options.bucket.grantReadWrite(this.props.project);
            }
            else {
                options.bucket.grantRead(this.props.project);
            }
        }
        if (this.props.project instanceof codebuild.Project) {
            this.props.project.bindToCodePipeline(scope, {
                artifactBucket: options.bucket,
            });
        }
        // if any of the inputs come from the BitBucketSourceAction
        // with codeBuildCloneOutput=true,
        // grant the Project's Role to use the connection
        for (const inputArtifact of this.actionProperties.inputs || []) {
            const connectionArn = inputArtifact.getMetadata(__1.BitBucketSourceAction._CONNECTION_ARN_PROPERTY);
            if (connectionArn) {
                this.props.project.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['codestar-connections:UseConnection'],
                    resources: [connectionArn],
                }));
            }
        }
        const configuration = {
            ProjectName: this.props.project.projectName,
            EnvironmentVariables: this.props.environmentVariables &&
                cdk.Stack.of(scope).toJsonString(codebuild.Project.serializeEnvVariables(this.props.environmentVariables)),
        };
        if ((this.actionProperties.inputs || []).length > 1) {
            // lazy, because the Artifact name might be generated lazily
            configuration.PrimarySource = cdk.Lazy.stringValue({ produce: () => this.props.input.artifactName });
        }
        return {
            configuration,
        };
    }
}
exports.CodeBuildAction = CodeBuildAction;
//# sourceMappingURL=data:application/json;base64,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