"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxGpuBuildImage = void 0;
const ecr = require("../../aws-ecr"); // Automatically re-written from '@aws-cdk/aws-ecr'
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const region_info_1 = require("../../region-info"); // Automatically re-written from '@aws-cdk/region-info'
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const project_1 = require("./project");
const mappingName = 'AwsDeepLearningContainersRepositoriesAccounts';
/**
 * A CodeBuild GPU image running Linux.
 *
 * This class has public constants that represent the most popular GPU images from AWS Deep Learning Containers.
 *
 * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
 */
class LinuxGpuBuildImage {
    constructor(repositoryName, tag, account) {
        this.repositoryName = repositoryName;
        this.account = account;
        this.type = 'LINUX_GPU_CONTAINER';
        this.defaultComputeType = project_1.ComputeType.LARGE;
        this.imagePullPrincipalType = project_1.ImagePullPrincipalType.SERVICE_ROLE;
        this.accountExpression = account !== null && account !== void 0 ? account : core.Fn.findInMap(mappingName, core.Aws.REGION, 'repositoryAccount');
        this.imageId = `${this.accountExpression}.dkr.ecr.${core.Aws.REGION}.${core.Aws.URL_SUFFIX}/${repositoryName}:${tag}`;
    }
    /**
     * Returns a Linux GPU build image from AWS Deep Learning Containers.
     *
     * @param repositoryName the name of the repository,
     *   for example "pytorch-inference"
     * @param tag the tag of the image, for example "1.5.0-gpu-py36-cu101-ubuntu16.04"
     * @param account the AWS account ID where the DLC repository for this region is hosted in.
     *   In many cases, the CDK can infer that for you, but for some newer region our information
     *   might be out of date; in that case, you can specify the region explicitly using this optional parameter
     * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
     */
    static awsDeepLearningContainersImage(repositoryName, tag, account) {
        return new LinuxGpuBuildImage(repositoryName, tag, account);
    }
    bind(scope, project, _options) {
        if (!this.account) {
            const scopeStack = core.Stack.of(scope);
            // Unfortunately, the account IDs of the DLC repositories are not the same in all regions.
            // Because of that, use a (singleton) Mapping to find the correct account
            if (!scopeStack.node.tryFindChild(mappingName)) {
                const mapping = {};
                // get the accounts from the region-info module
                const region2Accounts = region_info_1.RegionInfo.regionMap(region_info_1.FactName.DLC_REPOSITORY_ACCOUNT);
                for (const [region, account] of Object.entries(region2Accounts)) {
                    mapping[region] = { repositoryAccount: account };
                }
                new core.CfnMapping(scopeStack, mappingName, { mapping });
            }
        }
        const repository = ecr.Repository.fromRepositoryAttributes(scope, 'AwsDlcRepositoryCodeBuild', {
            repositoryName: this.repositoryName,
            repositoryArn: ecr.Repository.arnForLocalRepository(this.repositoryName, scope, this.accountExpression),
        });
        repository.grantPull(project);
        return {};
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== project_1.ComputeType.LARGE) {
            ret.push(`GPU images only support ComputeType '${project_1.ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxGpuBuildImage = LinuxGpuBuildImage;
/** Tensorflow 1.14.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_14_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.14.0-gpu-py36-cu100-ubuntu16.04');
/** Tensorflow 1.15.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.0-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 1.15.2 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.2-gpu-py37-cu100-ubuntu18.04');
/** Tensorflow 1.15.2 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '1.15.2-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 2.0.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.0-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 2.0.1 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.1-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 2.1.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/** Tensorflow 2.1.0 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/** Tensorflow 2.2.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_2_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.2.0-gpu-py37-cu101-ubuntu18.04');
/** PyTorch 1.2.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_2_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.2.0-gpu-py36-cu100-ubuntu16.04');
/** PyTorch 1.3.1 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_3_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.3.1-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.4.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.4.0 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.5.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.5.0 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/** MXNet 1.4.1 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_MXNET_1_4_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.4.1-gpu-py36-cu100-ubuntu16.04');
/** MXNet 1.6.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_MXNET_1_6_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.6.0-gpu-py36-cu101-ubuntu16.04');
//# sourceMappingURL=data:application/json;base64,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