"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MathExpression = exports.Metric = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const alarm_1 = require("./alarm");
const metric_util_1 = require("./private/metric-util");
const statistic_1 = require("./private/statistic");
/**
 * A metric emitted by a service
 *
 * The metric is a combination of a metric identifier (namespace, name and dimensions)
 * and an aggregation function (statistic, period and unit).
 *
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * Metric is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class Metric {
    constructor(props) {
        this.period = props.period || cdk.Duration.minutes(5);
        const periodSec = this.period.toSeconds();
        if (periodSec !== 1 && periodSec !== 5 && periodSec !== 10 && periodSec !== 30 && periodSec % 60 !== 0) {
            throw new Error(`'period' must be 1, 5, 10, 30, or a multiple of 60 seconds, received ${periodSec}`);
        }
        this.dimensions = props.dimensions;
        this.namespace = props.namespace;
        this.metricName = props.metricName;
        // Try parsing, this will throw if it's not a valid stat
        this.statistic = statistic_1.normalizeStatistic(props.statistic || 'Average');
        this.label = props.label;
        this.color = props.color;
        this.unit = props.unit;
        this.account = props.account;
        this.region = props.region;
    }
    /**
     * Grant permissions to the given identity to write metrics.
     *
     * @param grantee The IAM identity to give permissions to.
     */
    static grantPutMetricData(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['cloudwatch:PutMetricData'],
            resourceArns: ['*'],
        });
    }
    /**
     * Return a copy of Metric `with` properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.statistic === undefined || props.statistic === this.statistic)
            && (props.unit === undefined || props.unit === this.unit)
            && (props.account === undefined || props.account === this.account)
            && (props.region === undefined || props.region === this.region)
            // For these we're not going to do deep equality, misses some opportunity for optimization
            // but that's okay.
            && (props.dimensions === undefined)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new Metric({
            dimensions: ifUndefined(props.dimensions, this.dimensions),
            namespace: this.namespace,
            metricName: this.metricName,
            period: ifUndefined(props.period, this.period),
            statistic: ifUndefined(props.statistic, this.statistic),
            unit: ifUndefined(props.unit, this.unit),
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            account: ifUndefined(props.account, this.account),
            region: ifUndefined(props.region, this.region),
        });
    }
    /**
     * Attach the metric object to the given construct scope
     *
     * Returns a Metric object that uses the account and region from the Stack
     * the given construct is defined in. If the metric is subsequently used
     * in a Dashboard or Alarm in a different Stack defined in a different
     * account or region, the appropriate 'region' and 'account' fields
     * will be added to it.
     *
     * If the scope we attach to is in an environment-agnostic stack,
     * nothing is done and the same Metric object is returned.
     */
    attachTo(scope) {
        const stack = cdk.Stack.of(scope);
        return this.with({
            region: cdk.Token.isUnresolved(stack.region) ? undefined : stack.region,
            account: cdk.Token.isUnresolved(stack.account) ? undefined : stack.account,
        });
    }
    toMetricConfig() {
        const dims = this.dimensionsAsList();
        return {
            metricStat: {
                dimensions: dims.length > 0 ? dims : undefined,
                namespace: this.namespace,
                metricName: this.metricName,
                period: this.period,
                statistic: this.statistic,
                unitFilter: this.unit,
                account: this.account,
                region: this.region,
            },
            renderingProperties: {
                color: this.color,
                label: this.label,
            },
        };
    }
    toAlarmConfig() {
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        const stat = statistic_1.parseStatistic(metricConfig.metricStat.statistic);
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: stat.type === 'simple' ? stat.statistic : undefined,
            extendedStatistic: stat.type === 'percentile' ? 'p' + stat.percentile : undefined,
            unit: this.unit,
        };
    }
    toGraphConfig() {
        var _a, _b, _c, _d;
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            renderingProperties: {
                period: metricConfig.metricStat.period.toSeconds(),
                stat: metricConfig.metricStat.statistic,
                color: asString((_a = metricConfig.renderingProperties) === null || _a === void 0 ? void 0 : _a.color),
                label: asString((_b = metricConfig.renderingProperties) === null || _b === void 0 ? void 0 : _b.label),
            },
            // deprecated properties for backwards compatibility
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: metricConfig.metricStat.statistic,
            color: asString((_c = metricConfig.renderingProperties) === null || _c === void 0 ? void 0 : _c.color),
            label: asString((_d = metricConfig.renderingProperties) === null || _d === void 0 ? void 0 : _d.label),
            unit: this.unit,
        };
    }
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                statistic: props.statistic,
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    toString() {
        return this.label || this.metricName;
    }
    /**
     * Return the dimensions of this Metric as a list of Dimension.
     */
    dimensionsAsList() {
        const dims = this.dimensions;
        if (dims === undefined) {
            return [];
        }
        const list = Object.keys(dims).sort().map(key => ({ name: key, value: dims[key] }));
        return list;
    }
}
exports.Metric = Metric;
function asString(x) {
    if (x === undefined) {
        return undefined;
    }
    if (typeof x !== 'string') {
        throw new Error(`Expected string, got ${x}`);
    }
    return x;
}
/**
 * A math expression built with metric(s) emitted by a service
 *
 * The math expression is a combination of an expression (x+y) and metrics to apply expression on.
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * MathExpression is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 */
class MathExpression {
    constructor(props) {
        this.period = props.period || cdk.Duration.minutes(5);
        this.expression = props.expression;
        this.usingMetrics = changeAllPeriods(props.usingMetrics, this.period);
        this.label = props.label;
        this.color = props.color;
        const invalidVariableNames = Object.keys(props.usingMetrics).filter(x => !validVariableName(x));
        if (invalidVariableNames.length > 0) {
            throw new Error(`Invalid variable names in expression: ${invalidVariableNames}. Must start with lowercase letter and only contain alphanumerics.`);
        }
        this.validateNoIdConflicts();
    }
    /**
     * Return a copy of Metric with properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     */
    with(props) {
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new MathExpression({
            expression: this.expression,
            usingMetrics: this.usingMetrics,
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            period: ifUndefined(props.period, this.period),
        });
    }
    toAlarmConfig() {
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    toGraphConfig() {
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    toMetricConfig() {
        return {
            mathExpression: {
                period: this.period.toSeconds(),
                expression: this.expression,
                usingMetrics: this.usingMetrics,
            },
            renderingProperties: {
                label: this.label,
                color: this.color,
            },
        };
    }
    /**
     * Make a new Alarm for this metric
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     */
    createAlarm(scope, id, props) {
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    toString() {
        return this.label || this.expression;
    }
    validateNoIdConflicts() {
        const seen = new Map();
        visit(this);
        function visit(metric) {
            metric_util_1.dispatchMetric(metric, {
                withStat() {
                    // Nothing
                },
                withExpression(expr) {
                    for (const [id, subMetric] of Object.entries(expr.usingMetrics)) {
                        const existing = seen.get(id);
                        if (existing && metric_util_1.metricKey(existing) !== metric_util_1.metricKey(subMetric)) {
                            throw new Error(`The ID '${id}' used for two metrics in the expression: '${subMetric}' and '${existing}'. Rename one.`);
                        }
                        seen.set(id, subMetric);
                        visit(subMetric);
                    }
                },
            });
        }
    }
}
exports.MathExpression = MathExpression;
const VALID_VARIABLE = new RegExp('^[a-z][a-zA-Z0-9_]*$');
function validVariableName(x) {
    return VALID_VARIABLE.test(x);
}
function ifUndefined(x, def) {
    if (x !== undefined) {
        return x;
    }
    return def;
}
/**
 * Change periods of all metrics in the map
 */
function changeAllPeriods(metrics, period) {
    const ret = {};
    for (const [id, metric] of Object.entries(metrics)) {
        ret[id] = changePeriod(metric, period);
    }
    return ret;
}
/**
 * Return a new metric object which is the same type as the input object, but with the period changed
 *
 * Relies on the fact that implementations of `IMetric` are also supposed to have
 * an implementation of `with` that accepts an argument called `period`. See `IModifiableMetric`.
 */
function changePeriod(metric, period) {
    if (isModifiableMetric(metric)) {
        return metric.with({ period });
    }
    throw new Error(`Metric object should also implement 'with': ${metric}`);
}
function isModifiableMetric(m) {
    return typeof m === 'object' && m !== null && !!m.with;
}
//# sourceMappingURL=data:application/json;base64,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