"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PredefinedMetric = exports.TargetTrackingScalingPolicy = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error('Exactly one of \'customMetric\' or \'predefinedMetric\' must be specified.');
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error('Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.');
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldown && props.scaleInCooldown.toSeconds(),
                scaleOutCooldown: props.scaleOutCooldown && props.scaleOutCooldown.toSeconds(),
                targetValue: props.targetValue,
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    if (c.statistic.startsWith('p')) {
        throw new Error(`Cannot use statistic '${c.statistic}' for Target Tracking: only 'Average', 'Minimum', 'Maximum', 'SampleCount', and 'Sum' are supported.`);
    }
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    /**
     * DYNAMODB_READ_CAPACITY_UTILIZATIO
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYNAMODB_READ_CAPACITY_UTILIZATION"] = "DynamoDBReadCapacityUtilization";
    /**
     * DYANMODB_WRITE_CAPACITY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYANMODB_WRITE_CAPACITY_UTILIZATION"] = "DynamoDBWriteCapacityUtilization";
    /**
     * ALB_REQUEST_COUNT_PER_TARGET
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
    /**
     * RDS_READER_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_CPU_UTILIZATION"] = "RDSReaderAverageCPUUtilization";
    /**
     * RDS_READER_AVERAGE_DATABASE_CONNECTIONS
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_DATABASE_CONNECTIONS"] = "RDSReaderAverageDatabaseConnections";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION"] = "EC2SpotFleetRequestAverageCPUUtilization";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN"] = "EC2SpotFleetRequestAverageNetworkIn";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT"] = "EC2SpotFleetRequestAverageNetworkOut";
    /**
     * SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE"] = "SageMakerVariantInvocationsPerInstance";
    /**
     * ECS_SERVICE_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_CPU_UTILIZATION"] = "ECSServiceAverageCPUUtilization";
    /**
     * ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION"] = "ECSServiceAverageMemoryUtilization";
    /**
     * LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION
     * @see https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics.html#monitoring-metrics-concurrency
     */
    PredefinedMetric["LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION"] = "LambdaProvisionedConcurrencyUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFyZ2V0LXRyYWNraW5nLXNjYWxpbmctcG9saWN5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGFyZ2V0LXRyYWNraW5nLXNjYWxpbmctcG9saWN5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLGtDQUFrQyxDQUFDLGdEQUFnRDtBQUNuRix5RkFBc0U7QUF3R3RFLE1BQWEsMkJBQTRCLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFLMUQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUF1QztRQUNqRixJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksS0FBSyxTQUFTLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsS0FBSyxTQUFTLENBQUMsRUFBRTtZQUMvRSxNQUFNLElBQUksS0FBSyxDQUFDLDRFQUE0RSxDQUFDLENBQUM7U0FDakc7UUFDRCxJQUFJLEtBQUssQ0FBQyxZQUFZLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLGNBQWMsRUFBRSxDQUFDLFVBQVUsRUFBRTtZQUN2RSxNQUFNLElBQUksS0FBSyxDQUFDLG9HQUFvRyxDQUFDLENBQUM7U0FDekg7UUFDRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLE1BQU0sUUFBUSxHQUFHLElBQUksbURBQWdCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwRCxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVE7WUFDbEQsVUFBVSxFQUFFLHVCQUF1QjtZQUNuQyxlQUFlLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxnQkFBZ0I7WUFDckQsd0NBQXdDLEVBQUU7Z0JBQ3RDLDZCQUE2QixFQUFFLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUM7Z0JBQ3JFLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDcEMsNkJBQTZCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUM7b0JBQ2xFLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7b0JBQzVDLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtpQkFDckMsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDYixlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWUsSUFBSSxLQUFLLENBQUMsZUFBZSxDQUFDLFNBQVMsRUFBRTtnQkFDM0UsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUU7Z0JBQzlFLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVzthQUNqQztTQUNKLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO0lBQ3pDLENBQUM7Q0FDSjtBQS9CRCxrRUErQkM7QUFDRCxTQUFTLGtCQUFrQixDQUFDLE1BQTJCO0lBQ25ELElBQUksQ0FBQyxNQUFNLEVBQUU7UUFDVCxPQUFPLFNBQVMsQ0FBQztLQUNwQjtJQUNELE1BQU0sQ0FBQyxHQUFHLE1BQU0sQ0FBQyxjQUFjLEVBQUUsQ0FBQyxVQUFXLENBQUM7SUFDOUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLHlCQUF5QixDQUFDLENBQUMsU0FBUyxzR0FBc0csQ0FBQyxDQUFDO0tBQy9KO0lBQ0QsT0FBTztRQUNILFVBQVUsRUFBRSxDQUFDLENBQUMsVUFBVTtRQUN4QixVQUFVLEVBQUUsQ0FBQyxDQUFDLFVBQVU7UUFDeEIsU0FBUyxFQUFFLENBQUMsQ0FBQyxTQUFTO1FBQ3RCLFNBQVMsRUFBRSxDQUFDLENBQUMsU0FBUztRQUN0QixJQUFJLEVBQUUsQ0FBQyxDQUFDLFVBQVU7S0FDckIsQ0FBQztBQUNOLENBQUM7QUFDRDs7R0FFRztBQUNILElBQVksZ0JBNkRYO0FBN0RELFdBQVksZ0JBQWdCO0lBQ3hCOzs7T0FHRztJQUNILDBGQUFzRSxDQUFBO0lBQ3RFOzs7T0FHRztJQUNILDRGQUF3RSxDQUFBO0lBQ3hFOzs7T0FHRztJQUNILDZFQUF5RCxDQUFBO0lBQ3pEOzs7T0FHRztJQUNILHlGQUFxRSxDQUFBO0lBQ3JFOzs7T0FHRztJQUNILG1HQUErRSxDQUFBO0lBQy9FOzs7T0FHRztJQUNILCtHQUEyRixDQUFBO0lBQzNGOzs7T0FHRztJQUNILHFHQUFpRixDQUFBO0lBQ2pGOzs7T0FHRztJQUNILHVHQUFtRixDQUFBO0lBQ25GOzs7T0FHRztJQUNILHlHQUFxRixDQUFBO0lBQ3JGOzs7T0FHRztJQUNILDJGQUF1RSxDQUFBO0lBQ3ZFOzs7T0FHRztJQUNILGlHQUE2RSxDQUFBO0lBQzdFOzs7T0FHRztJQUNILDBHQUFzRixDQUFBO0FBQzFGLENBQUMsRUE3RFcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUE2RDNCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2xvdWR3YXRjaCBmcm9tIFwiLi4vLi4vYXdzLWNsb3Vkd2F0Y2hcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJ1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQ2ZuU2NhbGluZ1BvbGljeSB9IGZyb20gJy4vYXBwbGljYXRpb25hdXRvc2NhbGluZy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSVNjYWxhYmxlVGFyZ2V0IH0gZnJvbSAnLi9zY2FsYWJsZS10YXJnZXQnO1xuLyoqXG4gKiBCYXNlIGludGVyZmFjZSBmb3IgdGFyZ2V0IHRyYWNraW5nIHByb3BzXG4gKlxuICogQ29udGFpbnMgdGhlIGF0dHJpYnV0ZXMgdGhhdCBhcmUgY29tbW9uIHRvIHRhcmdldCB0cmFja2luZyBwb2xpY2llcyxcbiAqIGV4Y2VwdCB0aGUgb25lcyByZWxhdGluZyB0byB0aGUgbWV0cmljIGFuZCB0byB0aGUgc2NhbGFibGUgdGFyZ2V0LlxuICpcbiAqIFRoaXMgaW50ZXJmYWNlIGlzIHJldXNlZCBieSBtb3JlIHNwZWNpZmljIHRhcmdldCB0cmFja2luZyBwcm9wcyBvYmplY3RzXG4gKiBpbiBvdGhlciBzZXJ2aWNlcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCYXNlVGFyZ2V0VHJhY2tpbmdQcm9wcyB7XG4gICAgLyoqXG4gICAgICogQSBuYW1lIGZvciB0aGUgc2NhbGluZyBwb2xpY3lcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQXV0b21hdGljYWxseSBnZW5lcmF0ZWQgbmFtZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBwb2xpY3lOYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEluZGljYXRlcyB3aGV0aGVyIHNjYWxlIGluIGJ5IHRoZSB0YXJnZXQgdHJhY2tpbmcgcG9saWN5IGlzIGRpc2FibGVkLlxuICAgICAqXG4gICAgICogSWYgdGhlIHZhbHVlIGlzIHRydWUsIHNjYWxlIGluIGlzIGRpc2FibGVkIGFuZCB0aGUgdGFyZ2V0IHRyYWNraW5nIHBvbGljeVxuICAgICAqIHdvbid0IHJlbW92ZSBjYXBhY2l0eSBmcm9tIHRoZSBzY2FsYWJsZSByZXNvdXJjZS4gT3RoZXJ3aXNlLCBzY2FsZSBpbiBpc1xuICAgICAqIGVuYWJsZWQgYW5kIHRoZSB0YXJnZXQgdHJhY2tpbmcgcG9saWN5IGNhbiByZW1vdmUgY2FwYWNpdHkgZnJvbSB0aGVcbiAgICAgKiBzY2FsYWJsZSByZXNvdXJjZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGlzYWJsZVNjYWxlSW4/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFBlcmlvZCBhZnRlciBhIHNjYWxlIGluIGFjdGl2aXR5IGNvbXBsZXRlcyBiZWZvcmUgYW5vdGhlciBzY2FsZSBpbiBhY3Rpdml0eSBjYW4gc3RhcnQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5zZWNvbmRzKDMwMCkgZm9yIHRoZSBmb2xsb3dpbmcgc2NhbGFibGUgdGFyZ2V0czogRUNTIHNlcnZpY2VzLFxuICAgICAqIFNwb3QgRmxlZXQgcmVxdWVzdHMsIEVNUiBjbHVzdGVycywgQXBwU3RyZWFtIDIuMCBmbGVldHMsIEF1cm9yYSBEQiBjbHVzdGVycyxcbiAgICAgKiBBbWF6b24gU2FnZU1ha2VyIGVuZHBvaW50IHZhcmlhbnRzLCBDdXN0b20gcmVzb3VyY2VzLiBGb3IgYWxsIG90aGVyIHNjYWxhYmxlXG4gICAgICogdGFyZ2V0cywgdGhlIGRlZmF1bHQgdmFsdWUgaXMgRHVyYXRpb24uc2Vjb25kcygwKTogRHluYW1vREIgdGFibGVzLCBEeW5hbW9EQlxuICAgICAqIGdsb2JhbCBzZWNvbmRhcnkgaW5kZXhlcywgQW1hem9uIENvbXByZWhlbmQgZG9jdW1lbnQgY2xhc3NpZmljYXRpb24gZW5kcG9pbnRzLFxuICAgICAqIExhbWJkYSBwcm92aXNpb25lZCBjb25jdXJyZW5jeVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNjYWxlSW5Db29sZG93bj86IGNkay5EdXJhdGlvbjtcbiAgICAvKipcbiAgICAgKiBQZXJpb2QgYWZ0ZXIgYSBzY2FsZSBvdXQgYWN0aXZpdHkgY29tcGxldGVzIGJlZm9yZSBhbm90aGVyIHNjYWxlIG91dCBhY3Rpdml0eSBjYW4gc3RhcnQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5zZWNvbmRzKDMwMCkgZm9yIHRoZSBmb2xsb3dpbmcgc2NhbGFibGUgdGFyZ2V0czogRUNTIHNlcnZpY2VzLFxuICAgICAqIFNwb3QgRmxlZXQgcmVxdWVzdHMsIEVNUiBjbHVzdGVycywgQXBwU3RyZWFtIDIuMCBmbGVldHMsIEF1cm9yYSBEQiBjbHVzdGVycyxcbiAgICAgKiBBbWF6b24gU2FnZU1ha2VyIGVuZHBvaW50IHZhcmlhbnRzLCBDdXN0b20gcmVzb3VyY2VzLiBGb3IgYWxsIG90aGVyIHNjYWxhYmxlXG4gICAgICogdGFyZ2V0cywgdGhlIGRlZmF1bHQgdmFsdWUgaXMgRHVyYXRpb24uc2Vjb25kcygwKTogRHluYW1vREIgdGFibGVzLCBEeW5hbW9EQlxuICAgICAqIGdsb2JhbCBzZWNvbmRhcnkgaW5kZXhlcywgQW1hem9uIENvbXByZWhlbmQgZG9jdW1lbnQgY2xhc3NpZmljYXRpb24gZW5kcG9pbnRzLFxuICAgICAqIExhbWJkYSBwcm92aXNpb25lZCBjb25jdXJyZW5jeVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNjYWxlT3V0Q29vbGRvd24/OiBjZGsuRHVyYXRpb247XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgVGFyZ2V0IFRyYWNraW5nIHBvbGljeSB0aGF0IGluY2x1ZGUgdGhlIG1ldHJpYyBidXQgZXhjbHVkZSB0aGUgdGFyZ2V0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQmFzaWNUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3lQcm9wcyBleHRlbmRzIEJhc2VUYXJnZXRUcmFja2luZ1Byb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgdGFyZ2V0IHZhbHVlIGZvciB0aGUgbWV0cmljLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRhcmdldFZhbHVlOiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogQSBwcmVkZWZpbmVkIG1ldHJpYyBmb3IgYXBwbGljYXRpb24gYXV0b3NjYWxpbmdcbiAgICAgKlxuICAgICAqIFRoZSBtZXRyaWMgbXVzdCB0cmFjayB1dGlsaXphdGlvbi4gU2NhbGluZyBvdXQgd2lsbCBoYXBwZW4gaWYgdGhlIG1ldHJpYyBpcyBoaWdoZXIgdGhhblxuICAgICAqIHRoZSB0YXJnZXQgdmFsdWUsIHNjYWxpbmcgaW4gd2lsbCBoYXBwZW4gaW4gdGhlIG1ldHJpYyBpcyBsb3dlciB0aGFuIHRoZSB0YXJnZXQgdmFsdWUuXG4gICAgICpcbiAgICAgKiBFeGFjdGx5IG9uZSBvZiBjdXN0b21NZXRyaWMgb3IgcHJlZGVmaW5lZE1ldHJpYyBtdXN0IGJlIHNwZWNpZmllZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gcHJlZGVmaW5lZCBtZXRyaWNzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByZWRlZmluZWRNZXRyaWM/OiBQcmVkZWZpbmVkTWV0cmljO1xuICAgIC8qKlxuICAgICAqIElkZW50aWZ5IHRoZSByZXNvdXJjZSBhc3NvY2lhdGVkIHdpdGggdGhlIG1ldHJpYyB0eXBlLlxuICAgICAqXG4gICAgICogT25seSB1c2VkIGZvciBwcmVkZWZpbmVkIG1ldHJpYyBBTEJSZXF1ZXN0Q291bnRQZXJUYXJnZXQuXG4gICAgICpcbiAgICAgKiBAZXhhbXBsZSBhcHAvPGxvYWQtYmFsYW5jZXItbmFtZT4vPGxvYWQtYmFsYW5jZXItaWQ+L3RhcmdldGdyb3VwLzx0YXJnZXQtZ3JvdXAtbmFtZT4vPHRhcmdldC1ncm91cC1pZD5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gcmVzb3VyY2UgbGFiZWwuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzb3VyY2VMYWJlbD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBBIGN1c3RvbSBtZXRyaWMgZm9yIGFwcGxpY2F0aW9uIGF1dG9zY2FsaW5nXG4gICAgICpcbiAgICAgKiBUaGUgbWV0cmljIG11c3QgdHJhY2sgdXRpbGl6YXRpb24uIFNjYWxpbmcgb3V0IHdpbGwgaGFwcGVuIGlmIHRoZSBtZXRyaWMgaXMgaGlnaGVyIHRoYW5cbiAgICAgKiB0aGUgdGFyZ2V0IHZhbHVlLCBzY2FsaW5nIGluIHdpbGwgaGFwcGVuIGluIHRoZSBtZXRyaWMgaXMgbG93ZXIgdGhhbiB0aGUgdGFyZ2V0IHZhbHVlLlxuICAgICAqXG4gICAgICogRXhhY3RseSBvbmUgb2YgY3VzdG9tTWV0cmljIG9yIHByZWRlZmluZWRNZXRyaWMgbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGN1c3RvbSBtZXRyaWMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgY3VzdG9tTWV0cmljPzogY2xvdWR3YXRjaC5JTWV0cmljO1xufVxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIGNvbmNyZXRlIFRhcmdldFRyYWNraW5nUG9saWN5XG4gKlxuICogQWRkcyB0aGUgc2NhbGluZ1RhcmdldC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3lQcm9wcyBleHRlbmRzIEJhc2ljVGFyZ2V0VHJhY2tpbmdTY2FsaW5nUG9saWN5UHJvcHMge1xuICAgIC8qXG4gICAgICogVGhlIHNjYWxhYmxlIHRhcmdldFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNjYWxpbmdUYXJnZXQ6IElTY2FsYWJsZVRhcmdldDtcbn1cbmV4cG9ydCBjbGFzcyBUYXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3kgZXh0ZW5kcyBjZGsuQ29uc3RydWN0IHtcbiAgICAvKipcbiAgICAgKiBBUk4gb2YgdGhlIHNjYWxpbmcgcG9saWN5XG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IHNjYWxpbmdQb2xpY3lBcm46IHN0cmluZztcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFRhcmdldFRyYWNraW5nU2NhbGluZ1BvbGljeVByb3BzKSB7XG4gICAgICAgIGlmICgocHJvcHMuY3VzdG9tTWV0cmljID09PSB1bmRlZmluZWQpID09PSAocHJvcHMucHJlZGVmaW5lZE1ldHJpYyA9PT0gdW5kZWZpbmVkKSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdFeGFjdGx5IG9uZSBvZiBcXCdjdXN0b21NZXRyaWNcXCcgb3IgXFwncHJlZGVmaW5lZE1ldHJpY1xcJyBtdXN0IGJlIHNwZWNpZmllZC4nKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMuY3VzdG9tTWV0cmljICYmICFwcm9wcy5jdXN0b21NZXRyaWMudG9NZXRyaWNDb25maWcoKS5tZXRyaWNTdGF0KSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgZGlyZWN0IG1ldHJpY3MgYXJlIHN1cHBvcnRlZCBmb3IgVGFyZ2V0IFRyYWNraW5nLiBVc2UgU3RlcCBTY2FsaW5nIG9yIHN1cHBseSBhIE1ldHJpYyBvYmplY3QuJyk7XG4gICAgICAgIH1cbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuU2NhbGluZ1BvbGljeSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBwb2xpY3lOYW1lOiBwcm9wcy5wb2xpY3lOYW1lIHx8IHRoaXMubm9kZS51bmlxdWVJZCxcbiAgICAgICAgICAgIHBvbGljeVR5cGU6ICdUYXJnZXRUcmFja2luZ1NjYWxpbmcnLFxuICAgICAgICAgICAgc2NhbGluZ1RhcmdldElkOiBwcm9wcy5zY2FsaW5nVGFyZ2V0LnNjYWxhYmxlVGFyZ2V0SWQsXG4gICAgICAgICAgICB0YXJnZXRUcmFja2luZ1NjYWxpbmdQb2xpY3lDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICAgICAgY3VzdG9taXplZE1ldHJpY1NwZWNpZmljYXRpb246IHJlbmRlckN1c3RvbU1ldHJpYyhwcm9wcy5jdXN0b21NZXRyaWMpLFxuICAgICAgICAgICAgICAgIGRpc2FibGVTY2FsZUluOiBwcm9wcy5kaXNhYmxlU2NhbGVJbixcbiAgICAgICAgICAgICAgICBwcmVkZWZpbmVkTWV0cmljU3BlY2lmaWNhdGlvbjogcHJvcHMucHJlZGVmaW5lZE1ldHJpYyAhPT0gdW5kZWZpbmVkID8ge1xuICAgICAgICAgICAgICAgICAgICBwcmVkZWZpbmVkTWV0cmljVHlwZTogcHJvcHMucHJlZGVmaW5lZE1ldHJpYyxcbiAgICAgICAgICAgICAgICAgICAgcmVzb3VyY2VMYWJlbDogcHJvcHMucmVzb3VyY2VMYWJlbCxcbiAgICAgICAgICAgICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgICAgIHNjYWxlSW5Db29sZG93bjogcHJvcHMuc2NhbGVJbkNvb2xkb3duICYmIHByb3BzLnNjYWxlSW5Db29sZG93bi50b1NlY29uZHMoKSxcbiAgICAgICAgICAgICAgICBzY2FsZU91dENvb2xkb3duOiBwcm9wcy5zY2FsZU91dENvb2xkb3duICYmIHByb3BzLnNjYWxlT3V0Q29vbGRvd24udG9TZWNvbmRzKCksXG4gICAgICAgICAgICAgICAgdGFyZ2V0VmFsdWU6IHByb3BzLnRhcmdldFZhbHVlLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuc2NhbGluZ1BvbGljeUFybiA9IHJlc291cmNlLnJlZjtcbiAgICB9XG59XG5mdW5jdGlvbiByZW5kZXJDdXN0b21NZXRyaWMobWV0cmljPzogY2xvdWR3YXRjaC5JTWV0cmljKTogQ2ZuU2NhbGluZ1BvbGljeS5DdXN0b21pemVkTWV0cmljU3BlY2lmaWNhdGlvblByb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoIW1ldHJpYykge1xuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICBjb25zdCBjID0gbWV0cmljLnRvTWV0cmljQ29uZmlnKCkubWV0cmljU3RhdCE7XG4gICAgaWYgKGMuc3RhdGlzdGljLnN0YXJ0c1dpdGgoJ3AnKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCB1c2Ugc3RhdGlzdGljICcke2Muc3RhdGlzdGljfScgZm9yIFRhcmdldCBUcmFja2luZzogb25seSAnQXZlcmFnZScsICdNaW5pbXVtJywgJ01heGltdW0nLCAnU2FtcGxlQ291bnQnLCBhbmQgJ1N1bScgYXJlIHN1cHBvcnRlZC5gKTtcbiAgICB9XG4gICAgcmV0dXJuIHtcbiAgICAgICAgZGltZW5zaW9uczogYy5kaW1lbnNpb25zLFxuICAgICAgICBtZXRyaWNOYW1lOiBjLm1ldHJpY05hbWUsXG4gICAgICAgIG5hbWVzcGFjZTogYy5uYW1lc3BhY2UsXG4gICAgICAgIHN0YXRpc3RpYzogYy5zdGF0aXN0aWMsXG4gICAgICAgIHVuaXQ6IGMudW5pdEZpbHRlcixcbiAgICB9O1xufVxuLyoqXG4gKiBPbmUgb2YgdGhlIHByZWRlZmluZWQgYXV0b3NjYWxpbmcgbWV0cmljc1xuICovXG5leHBvcnQgZW51bSBQcmVkZWZpbmVkTWV0cmljIHtcbiAgICAvKipcbiAgICAgKiBEWU5BTU9EQl9SRUFEX0NBUEFDSVRZX1VUSUxJWkFUSU9cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hdXRvc2NhbGluZy9hcHBsaWNhdGlvbi9BUElSZWZlcmVuY2UvQVBJX1ByZWRlZmluZWRNZXRyaWNTcGVjaWZpY2F0aW9uLmh0bWxcbiAgICAgKi9cbiAgICBEWU5BTU9EQl9SRUFEX0NBUEFDSVRZX1VUSUxJWkFUSU9OID0gJ0R5bmFtb0RCUmVhZENhcGFjaXR5VXRpbGl6YXRpb24nLFxuICAgIC8qKlxuICAgICAqIERZQU5NT0RCX1dSSVRFX0NBUEFDSVRZX1VUSUxJWkFUSU9OXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXV0b3NjYWxpbmcvYXBwbGljYXRpb24vQVBJUmVmZXJlbmNlL0FQSV9QcmVkZWZpbmVkTWV0cmljU3BlY2lmaWNhdGlvbi5odG1sXG4gICAgICovXG4gICAgRFlBTk1PREJfV1JJVEVfQ0FQQUNJVFlfVVRJTElaQVRJT04gPSAnRHluYW1vREJXcml0ZUNhcGFjaXR5VXRpbGl6YXRpb24nLFxuICAgIC8qKlxuICAgICAqIEFMQl9SRVFVRVNUX0NPVU5UX1BFUl9UQVJHRVRcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hdXRvc2NhbGluZy9hcHBsaWNhdGlvbi9BUElSZWZlcmVuY2UvQVBJX1ByZWRlZmluZWRNZXRyaWNTcGVjaWZpY2F0aW9uLmh0bWxcbiAgICAgKi9cbiAgICBBTEJfUkVRVUVTVF9DT1VOVF9QRVJfVEFSR0VUID0gJ0FMQlJlcXVlc3RDb3VudFBlclRhcmdldCcsXG4gICAgLyoqXG4gICAgICogUkRTX1JFQURFUl9BVkVSQUdFX0NQVV9VVElMSVpBVElPTlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2FwcGxpY2F0aW9uL0FQSVJlZmVyZW5jZS9BUElfUHJlZGVmaW5lZE1ldHJpY1NwZWNpZmljYXRpb24uaHRtbFxuICAgICAqL1xuICAgIFJEU19SRUFERVJfQVZFUkFHRV9DUFVfVVRJTElaQVRJT04gPSAnUkRTUmVhZGVyQXZlcmFnZUNQVVV0aWxpemF0aW9uJyxcbiAgICAvKipcbiAgICAgKiBSRFNfUkVBREVSX0FWRVJBR0VfREFUQUJBU0VfQ09OTkVDVElPTlNcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hdXRvc2NhbGluZy9hcHBsaWNhdGlvbi9BUElSZWZlcmVuY2UvQVBJX1ByZWRlZmluZWRNZXRyaWNTcGVjaWZpY2F0aW9uLmh0bWxcbiAgICAgKi9cbiAgICBSRFNfUkVBREVSX0FWRVJBR0VfREFUQUJBU0VfQ09OTkVDVElPTlMgPSAnUkRTUmVhZGVyQXZlcmFnZURhdGFiYXNlQ29ubmVjdGlvbnMnLFxuICAgIC8qKlxuICAgICAqIEVDMl9TUE9UX0ZMRUVUX1JFUVVFU1RfQVZFUkFHRV9DUFVfVVRJTElaQVRJT05cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hdXRvc2NhbGluZy9hcHBsaWNhdGlvbi9BUElSZWZlcmVuY2UvQVBJX1ByZWRlZmluZWRNZXRyaWNTcGVjaWZpY2F0aW9uLmh0bWxcbiAgICAgKi9cbiAgICBFQzJfU1BPVF9GTEVFVF9SRVFVRVNUX0FWRVJBR0VfQ1BVX1VUSUxJWkFUSU9OID0gJ0VDMlNwb3RGbGVldFJlcXVlc3RBdmVyYWdlQ1BVVXRpbGl6YXRpb24nLFxuICAgIC8qKlxuICAgICAqIEVDMl9TUE9UX0ZMRUVUX1JFUVVFU1RfQVZFUkFHRV9ORVRXT1JLX0lOXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXV0b3NjYWxpbmcvYXBwbGljYXRpb24vQVBJUmVmZXJlbmNlL0FQSV9QcmVkZWZpbmVkTWV0cmljU3BlY2lmaWNhdGlvbi5odG1sXG4gICAgICovXG4gICAgRUMyX1NQT1RfRkxFRVRfUkVRVUVTVF9BVkVSQUdFX05FVFdPUktfSU4gPSAnRUMyU3BvdEZsZWV0UmVxdWVzdEF2ZXJhZ2VOZXR3b3JrSW4nLFxuICAgIC8qKlxuICAgICAqIEVDMl9TUE9UX0ZMRUVUX1JFUVVFU1RfQVZFUkFHRV9ORVRXT1JLX09VVFxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2FwcGxpY2F0aW9uL0FQSVJlZmVyZW5jZS9BUElfUHJlZGVmaW5lZE1ldHJpY1NwZWNpZmljYXRpb24uaHRtbFxuICAgICAqL1xuICAgIEVDMl9TUE9UX0ZMRUVUX1JFUVVFU1RfQVZFUkFHRV9ORVRXT1JLX09VVCA9ICdFQzJTcG90RmxlZXRSZXF1ZXN0QXZlcmFnZU5ldHdvcmtPdXQnLFxuICAgIC8qKlxuICAgICAqIFNBR0VNQUtFUl9WQVJJQU5UX0lOVk9DQVRJT05TX1BFUl9JTlNUQU5DRVxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2FwcGxpY2F0aW9uL0FQSVJlZmVyZW5jZS9BUElfUHJlZGVmaW5lZE1ldHJpY1NwZWNpZmljYXRpb24uaHRtbFxuICAgICAqL1xuICAgIFNBR0VNQUtFUl9WQVJJQU5UX0lOVk9DQVRJT05TX1BFUl9JTlNUQU5DRSA9ICdTYWdlTWFrZXJWYXJpYW50SW52b2NhdGlvbnNQZXJJbnN0YW5jZScsXG4gICAgLyoqXG4gICAgICogRUNTX1NFUlZJQ0VfQVZFUkFHRV9DUFVfVVRJTElaQVRJT05cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hdXRvc2NhbGluZy9hcHBsaWNhdGlvbi9BUElSZWZlcmVuY2UvQVBJX1ByZWRlZmluZWRNZXRyaWNTcGVjaWZpY2F0aW9uLmh0bWxcbiAgICAgKi9cbiAgICBFQ1NfU0VSVklDRV9BVkVSQUdFX0NQVV9VVElMSVpBVElPTiA9ICdFQ1NTZXJ2aWNlQXZlcmFnZUNQVVV0aWxpemF0aW9uJyxcbiAgICAvKipcbiAgICAgKiBFQ1NfU0VSVklDRV9BVkVSQUdFX01FTU9SWV9VVElMSVpBVElPTlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2FwcGxpY2F0aW9uL0FQSVJlZmVyZW5jZS9BUElfUHJlZGVmaW5lZE1ldHJpY1NwZWNpZmljYXRpb24uaHRtbFxuICAgICAqL1xuICAgIEVDU19TRVJWSUNFX0FWRVJBR0VfTUVNT1JZX1VUSUxJWkFUSU9OID0gJ0VDU1NlcnZpY2VBdmVyYWdlTWVtb3J5VXRpbGl6YXRpb24nLFxuICAgIC8qKlxuICAgICAqIExBTUJEQV9QUk9WSVNJT05FRF9DT05DVVJSRU5DWV9VVElMSVpBVElPTlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhbWJkYS9sYXRlc3QvZGcvbW9uaXRvcmluZy1tZXRyaWNzLmh0bWwjbW9uaXRvcmluZy1tZXRyaWNzLWNvbmN1cnJlbmN5XG4gICAgICovXG4gICAgTEFNQkRBX1BST1ZJU0lPTkVEX0NPTkNVUlJFTkNZX1VUSUxJWkFUSU9OID0gJ0xhbWJkYVByb3Zpc2lvbmVkQ29uY3VycmVuY3lVdGlsaXphdGlvbidcbn1cbiJdfQ==