"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseScalableAttribute = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const scalable_target_1 = require("./scalable-target");
/**
 * Represent an attribute for which autoscaling can be configured
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *   Step Scaling, so the Dynamo subclass won't expose this method).
 */
class BaseScalableAttribute extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity,
        });
    }
    /**
     * Scale out or in based on time
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,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