import { Construct, Resource } from "../../core";
import { IApiKey } from './api-key';
import { Method } from './method';
import { IRestApi } from './restapi';
import { Stage } from './stage';
/**
 * Container for defining throttling parameters to API stages or methods.
 * @link https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-request-throttling.html
 */
export interface ThrottleSettings {
    /**
     * The API request steady-state rate limit (average requests per second over an extended period of time)
     * @default none
     */
    readonly rateLimit?: number;
    /**
     * The maximum API request rate limit over a time ranging from one to a few seconds.
     * @default none
     */
    readonly burstLimit?: number;
}
/**
 * Time period for which quota settings apply.
 */
export declare enum Period {
    DAY = "DAY",
    WEEK = "WEEK",
    MONTH = "MONTH"
}
/**
 * Specifies the maximum number of requests that clients can make to API Gateway APIs.
 */
export interface QuotaSettings {
    /**
     * The maximum number of requests that users can make within the specified time period.
     * @default none
     */
    readonly limit?: number;
    /**
     * For the initial time period, the number of requests to subtract from the specified limit.
     * @default none
     */
    readonly offset?: number;
    /**
     * The time period for which the maximum limit of requests applies.
     * @default none
     */
    readonly period?: Period;
}
/**
 * Represents per-method throttling for a resource.
 */
export interface ThrottlingPerMethod {
    /**
     * [disable-awslint:ref-via-interface]
     * The method for which you specify the throttling settings.
     * @default none
     */
    readonly method: Method;
    /**
     * Specifies the overall request rate (average requests per second) and burst capacity.
     * @default none
     */
    readonly throttle: ThrottleSettings;
}
/**
 * Represents the API stages that a usage plan applies to.
 */
export interface UsagePlanPerApiStage {
    /**
     * @default none
     */
    readonly api?: IRestApi;
    /**
     *
     * [disable-awslint:ref-via-interface]
     * @default none
     */
    readonly stage?: Stage;
    /**
     * @default none
     */
    readonly throttle?: ThrottlingPerMethod[];
}
export interface UsagePlanProps {
    /**
     * API Stages to be associated with the usage plan.
     * @default none
     */
    readonly apiStages?: UsagePlanPerApiStage[];
    /**
     * Represents usage plan purpose.
     * @default none
     */
    readonly description?: string;
    /**
     * Number of requests clients can make in a given time period.
     * @default none
     */
    readonly quota?: QuotaSettings;
    /**
     * Overall throttle settings for the API.
     * @default none
     */
    readonly throttle?: ThrottleSettings;
    /**
     * Name for this usage plan.
     * @default none
     */
    readonly name?: string;
    /**
     * ApiKey to be associated with the usage plan.
     * @default none
     */
    readonly apiKey?: IApiKey;
}
export declare class UsagePlan extends Resource {
    /**
     * @attribute
     */
    readonly usagePlanId: string;
    private readonly apiStages;
    constructor(scope: Construct, id: string, props?: UsagePlanProps);
    /**
     * Adds an ApiKey.
     *
     * @param apiKey
     */
    addApiKey(apiKey: IApiKey): void;
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage: UsagePlanPerApiStage): void;
    /**
     *
     * @param props
     */
    private renderApiStages;
    private createStage;
    private renderQuota;
    private renderThrottle;
    private renderThrottlePerMethod;
}
