// Copyright (c) Mito
import React, { useEffect, useState } from 'react';
import { isMitoError } from '../../../utils/errors';
import TextButton from '../../elements/TextButton';
import ConfigureIcon from '../../icons/ConfigureIcon';
import Col from '../../layout/Col';
import Row from '../../layout/Row';
import DefaultTaskpane from '../DefaultTaskpane/DefaultTaskpane';
import DefaultTaskpaneBody from '../DefaultTaskpane/DefaultTaskpaneBody';
import DefaultTaskpaneFooter from '../DefaultTaskpane/DefaultTaskpaneFooter';
import DefaultTaskpaneHeader from '../DefaultTaskpane/DefaultTaskpaneHeader';
import CSVImport from './CSVImport';
import FileBrowser from './FileBrowser';
import { getElementsToDisplay, getFileEnding, getImportButtonStatus, isExcelFile } from './importUtils';
import XLSXImport from './XLSXImport';
function ImportTaskpane(props) {
    const [importState, setImportState] = useState({
        pathContents: {
            path_parts: props.currPathParts,
            elements: []
        },
        sort: 'last_modified_descending',
        searchString: '',
        selectedElementIndex: -1,
        loadingFolder: false,
        loadingImport: false
    });
    // If the file being imported is an XLSX, we need additional configuration
    // and so we use an import wizard for help
    const [fileForImportWizard, setFileForImportWizard] = useState(undefined);
    // It is very convenient to have the full joined path for the file, so this state and the 
    // effect below it make it possible to access this easily
    const [fullFileNameForImportWizard, setFullFileNameForImportWizard] = useState(undefined);
    // Track if there has been an error
    const [importError, setImportError] = useState(undefined);
    useEffect(() => {
        const getFullFileNameForImportWizard = async (fileForImportWizard) => {
            const finalPath = [...props.currPathParts];
            finalPath.push(fileForImportWizard);
            const fullFileName = await props.mitoAPI.getPathJoined(finalPath);
            setFullFileNameForImportWizard(fullFileName);
        };
        if (fileForImportWizard !== undefined) {
            void getFullFileNameForImportWizard(fileForImportWizard.name);
        }
        else {
            setFullFileNameForImportWizard(undefined);
        }
    }, [fileForImportWizard]);
    // We make sure to get the elements that are displayed and use the index on that to get the correct element
    const selectedElement = getElementsToDisplay(importState)[importState.selectedElementIndex];
    /*
        Any time the current path changes, we update
        the files that are displayed
    */
    useEffect(() => {
        // When the current path changes, we reload the path contents
        void loadPathContents(props.currPathParts);
        // We also unselect anything that might be selected
        setImportState(prevImportState => {
            return Object.assign(Object.assign({}, prevImportState), { selectedElementIndex: -1 });
        });
        // Log how long the path is
        void props.mitoAPI.log('curr_path_changed', { 'path_parts_length': props.currPathParts.length });
    }, [props.currPathParts]);
    /*
        Any time the selected element changes we log the file
        ending (or none, if it has none).
    */
    useEffect(() => {
        let selectedElementName = '';
        if (selectedElement === undefined) {
            selectedElementName = 'undefined';
        }
        else if (selectedElement.isDirectory) {
            selectedElementName = 'directory';
        }
        else {
            const fileEnding = getFileEnding(selectedElement.name);
            if (fileEnding !== undefined) {
                selectedElementName = fileEnding;
            }
            else {
                selectedElementName = 'No File Ending';
            }
        }
        void props.mitoAPI.log('selected_element_changed', { 'selected_element': selectedElementName });
    }, [selectedElement]);
    // Loads the path data from the API and sets it for the file browser
    async function loadPathContents(currPathParts) {
        setImportState(prevImportState => {
            return Object.assign(Object.assign({}, prevImportState), { loadingFolder: true });
        });
        const _pathContents = await props.mitoAPI.getPathContents(currPathParts);
        if (_pathContents) {
            setImportState(prevImportState => {
                return Object.assign(Object.assign({}, prevImportState), { pathContents: _pathContents, loadingFolder: false });
            });
        }
        else {
            setImportState(prevImportState => {
                return Object.assign(Object.assign({}, prevImportState), { loadingFolder: false });
            });
        }
    }
    async function importElement(element) {
        const importButtonStatus = getImportButtonStatus(element, props.userProfile.excelImportEnabled, importState.loadingImport);
        // Quit early if the selected thing is not importable, or if there
        // is nothing even selected
        if (importButtonStatus.disabled || element === undefined) {
            return;
        }
        if (isExcelFile(element)) {
            setFileForImportWizard(element);
            return;
        }
        // Do the actual import
        const finalPath = [...props.currPathParts];
        finalPath.push(element.name);
        const joinedPath = await props.mitoAPI.getPathJoined(finalPath);
        if (joinedPath === undefined) {
            return;
        }
        // And then actually import it
        setImportState(prevImportState => {
            return Object.assign(Object.assign({}, prevImportState), { loadingImport: true });
        });
        const possibleMitoError = await props.mitoAPI.editSimpleImport([joinedPath]);
        if (isMitoError(possibleMitoError)) {
            // If this an error, then we open the CSV config 
            setImportError(possibleMitoError);
            setFileForImportWizard(element);
        }
        setImportState(prevImportState => {
            return Object.assign(Object.assign({}, prevImportState), { loadingImport: false });
        });
    }
    const importButtonStatus = getImportButtonStatus(selectedElement, props.userProfile.excelImportEnabled, importState.loadingImport);
    // Check both the file and the full file name so that 
    // the screen does not flash when the back button is pressed
    // in the import wizard
    if (fileForImportWizard !== undefined && fullFileNameForImportWizard !== undefined && isExcelFile(fileForImportWizard)) {
        return (React.createElement(XLSXImport, { mitoAPI: props.mitoAPI, analysisData: props.analysisData, fileName: fullFileNameForImportWizard, fileForImportWizard: fileForImportWizard, setFileForImportWizard: setFileForImportWizard, setImportState: setImportState, setUIState: props.setUIState, importState: importState }));
    }
    else if (fileForImportWizard !== undefined && fullFileNameForImportWizard !== undefined) {
        return (React.createElement(CSVImport, { mitoAPI: props.mitoAPI, analysisData: props.analysisData, fileName: fullFileNameForImportWizard, fileForImportWizard: fileForImportWizard, setFileForImportWizard: setFileForImportWizard, setImportState: setImportState, setUIState: props.setUIState, importState: importState, error: importError, setError: setImportError }));
    }
    return (React.createElement(DefaultTaskpane, null,
        React.createElement(DefaultTaskpaneHeader, { header: 'Import Files', setUIState: props.setUIState }),
        React.createElement(DefaultTaskpaneBody, { noScroll: true },
            React.createElement(FileBrowser, { mitoAPI: props.mitoAPI, setCurrPathParts: props.setCurrPathParts, setUIState: props.setUIState, importState: importState, setImportState: setImportState, importElement: importElement, userProfile: props.userProfile })),
        React.createElement(DefaultTaskpaneFooter, null,
            React.createElement(Row, { justify: 'space-between' },
                !importButtonStatus.disabled && !isExcelFile(selectedElement) &&
                    React.createElement(Col, null,
                        React.createElement(TextButton, { variant: 'light', width: 'small', onClick: () => {
                                setImportError(undefined);
                                setFileForImportWizard(selectedElement);
                            }, disabled: importButtonStatus.disabled },
                            React.createElement(Row, { suppressTopBottomMargin: true, justify: 'space-between', align: 'center' },
                                React.createElement(ConfigureIcon, null),
                                React.createElement("p", { className: 'ml-2px' }, "Configure")))),
                React.createElement(Col, { span: !importButtonStatus.disabled && !isExcelFile(selectedElement) ? 18 : 24 },
                    React.createElement(TextButton, { variant: 'dark', width: 'block', onClick: () => {
                            void importElement(selectedElement);
                        }, disabled: importButtonStatus.disabled }, importButtonStatus.buttonText))))));
}
export default ImportTaskpane;
//# sourceMappingURL=ImportTaskpane.js.map