// Copyright (c) Mito
// Distributed under the terms of the Modified BSD License.
import React, { useState } from 'react';
import DefaultTaskpane from '../DefaultTaskpane/DefaultTaskpane';
// Import 
import TextButton from '../../elements/TextButton';
import Row from '../../layout/Row';
import Select from '../../elements/Select';
import DropdownItem from '../../elements/DropdownItem';
import { useDebouncedEffect } from '../../../hooks/useDebouncedEffect';
import LoadingDots from '../../elements/LoadingDots';
import ExcelDownloadConfigSection from './ExcelDownloadConfigSection';
import CSVDownloadConfigSection from './CSVDownloadConfigSection';
import DefaultTaskpaneHeader from '../DefaultTaskpane/DefaultTaskpaneHeader';
import DefaultTaskpaneBody from '../DefaultTaskpane/DefaultTaskpaneBody';
import DefaultEmptyTaskpane from '../DefaultTaskpane/DefaultEmptyTaskpane';
import DefaultTaskpaneFooter from '../DefaultTaskpane/DefaultTaskpaneFooter';
/*
    A taskpane that allows a user to download their current sheet.

    It does this by:
    1. Getting a string representation of the sheet through the api
    2. Encoding that as a file
    3. Allowing the user to download that file

    To see more about this process, read documentation here:
    https://blog.logrocket.com/programmatic-file-downloads-in-the-browser-9a5186298d5c/
*/
const DownloadTaskpane = (props) => {
    var _a;
    /*
        Store the columnIDs that the user has adding formatting to through the export taskpane so when
        the user adds a new column to be formatted, it appears at the top of the list of formatted
        columns, no matter the column's position in the sheet. Making it easier to find
        the column in the list.
    */
    const [newlyFormattedColumns, setNewlyFormattedColumns] = useState(() => {
        const newlyFormattedColumnsInitial = {};
        props.sheetDataArray.forEach((_, idx) => {
            newlyFormattedColumnsInitial[idx] = [];
        });
        return newlyFormattedColumnsInitial;
    });
    // The string that stores the file that actually should be downloaded
    const [exportString, setExportString] = useState('');
    const emptySheet = props.sheetDataArray.length === 0;
    const numRows = (_a = props.sheetDataArray[props.selectedSheetIndex]) === null || _a === void 0 ? void 0 : _a.numRows;
    const loadExport = async () => {
        // Don't try and load data if the sheet is empty
        if (emptySheet) {
            return;
        }
        if (props.uiState.exportConfiguration.exportType === 'csv') {
            const csvString = await props.mitoAPI.getDataframeAsCSV(props.selectedSheetIndex);
            setExportString(csvString);
        }
        else if (props.uiState.exportConfiguration.exportType === 'excel') {
            const excelString = await props.mitoAPI.getDataframesAsExcel(props.uiState.exportConfiguration.sheetIndexes);
            setExportString(excelString);
        }
    };
    // Async load in the data from the mitoAPI
    useDebouncedEffect(() => {
        setExportString('');
        void loadExport();
    }, [props.uiState.exportConfiguration, props.selectedSheetIndex, props.sheetDataArray], 500);
    const onDownload = () => {
        void props.mitoAPI.log('button_download_log_event', {
            sheet_index: props.selectedSheetIndex,
            export_type: props.uiState.exportConfiguration.exportType
        });
    };
    if (emptySheet) {
        return React.createElement(DefaultEmptyTaskpane, { setUIState: props.setUIState });
    }
    let exportHRef = '';
    let exportName = '';
    if (props.uiState.exportConfiguration.exportType === 'csv') {
        exportHRef = URL.createObjectURL(new Blob([exportString], { type: 'text/csv' }));
        exportName = 'MitoExport.csv';
    }
    else if (props.uiState.exportConfiguration.exportType === 'excel') {
        exportHRef = URL.createObjectURL(new Blob(
        /*
            First, we convert the export string out of base 64 encoding,
            and the convert it back into bytes
        */
        [Uint8Array.from(window.atob(exportString), c => c.charCodeAt(0))], { type: 'text/csv' } // TODO: for some reason, text/csv works fine here
        ));
        exportName = 'MitoExport.xlsx';
    }
    return (React.createElement(DefaultTaskpane, null,
        React.createElement(DefaultTaskpaneHeader, { header: 'Download', setUIState: props.setUIState }),
        React.createElement(DefaultTaskpaneBody, null,
            React.createElement("div", null,
                React.createElement(Row, { justify: 'space-between', align: 'center' },
                    React.createElement("p", { className: 'text-header-3' }, "Export Type"),
                    React.createElement(Select, { width: 'medium', value: props.uiState.exportConfiguration.exportType, onChange: (newExportType) => {
                            setExportString('');
                            props.setUIState(prevUIState => {
                                if (newExportType === 'csv') {
                                    return Object.assign(Object.assign({}, prevUIState), { exportConfiguration: { exportType: 'csv' } });
                                }
                                else {
                                    return Object.assign(Object.assign({}, prevUIState), { exportConfiguration: { exportType: 'excel', sheetIndexes: [props.selectedSheetIndex] } });
                                }
                            });
                        } },
                        React.createElement(DropdownItem, { title: 'csv' }),
                        React.createElement(DropdownItem, { title: 'excel', subtext: numRows > 1048576
                                ? `An Excel file holds at most 1,048,576 rows, but there are ${numRows} rows in this dataframe. We'll export the first 1,048,576 rows, but this may take several minutes.`
                                : `Due to Python limitations, Excel export can be slower than CSV export.` }))),
                props.uiState.exportConfiguration.exportType === 'excel' &&
                    React.createElement(ExcelDownloadConfigSection, { dfNames: props.dfNames, mitoAPI: props.mitoAPI, userProfile: props.userProfile, sheetDataArray: props.sheetDataArray, exportState: props.uiState.exportConfiguration, setUIState: props.setUIState, newlyFormattedColumns: newlyFormattedColumns, setNewlyFormattedColumns: setNewlyFormattedColumns }),
                props.uiState.exportConfiguration.exportType === 'csv' &&
                    React.createElement(CSVDownloadConfigSection, { sheetDataArray: props.sheetDataArray, mitoAPI: props.mitoAPI, selectedSheetIndex: props.selectedSheetIndex, setUIState: props.setUIState }))),
        React.createElement(DefaultTaskpaneFooter, null,
            React.createElement(TextButton, { variant: 'dark', width: 'block', disabled: exportString === '', href: exportHRef, download: exportName, onClick: onDownload }, exportString === '' ? (React.createElement(React.Fragment, null,
                "Preparing data for download ",
                React.createElement(LoadingDots, null))) : `Download ${props.uiState.exportConfiguration.exportType === 'csv' ? 'CSV file' : 'Excel workbook'}`))));
};
export default DownloadTaskpane;
//# sourceMappingURL=DownloadTaskpane.js.map