import React, { useState } from "react";
import useLiveUpdatingParams from "../../../hooks/useLiveUpdatingParams";
import { StepType } from "../../../types";
import DropdownButton from "../../elements/DropdownButton";
import DropdownItem from "../../elements/DropdownItem";
import Select from "../../elements/Select";
import SelectAndXIconCard from "../../elements/SelectAndXIconCard";
import Toggle from "../../elements/Toggle";
import Tooltip from "../../elements/Tooltip";
import Col from "../../layout/Col";
import Row from "../../layout/Row";
import DefaultEmptyTaskpane from "../DefaultTaskpane/DefaultEmptyTaskpane";
import DefaultTaskpane from "../DefaultTaskpane/DefaultTaskpane";
import DefaultTaskpaneBody from "../DefaultTaskpane/DefaultTaskpaneBody";
import DefaultTaskpaneHeader from "../DefaultTaskpane/DefaultTaskpaneHeader";
import { getFirstCharactersOfColumnHeaders } from "../../../utils/columnHeaders";
/*
    Constructs a message that says if all the column headers in a sheet are included in the concatanated sheet.
    If any column headers are not included, it reports them to the user.
*/
const getColumnHeadersIncludedMessage = (notIncludedColumnsArray, arrIndex) => {
    if (notIncludedColumnsArray[arrIndex].length === 0) {
        return (React.createElement("p", null, "\u2713 All columns are included in the concatenated sheet."));
    }
    const [columnHeadersString, numOtherColumnHeaders] = getFirstCharactersOfColumnHeaders(notIncludedColumnsArray[arrIndex], 25);
    if (numOtherColumnHeaders === 0) {
        return (React.createElement("p", null,
            "Columns ",
            React.createElement("span", { className: 'text-color-gray-important' }, columnHeadersString),
            " are not included."));
    }
    else {
        return (React.createElement("p", null,
            "Columns ",
            React.createElement("span", { className: 'text-color-gray-important' }, columnHeadersString),
            " and ",
            React.createElement("span", { className: 'text-color-gray-important' }, numOtherColumnHeaders),
            " others are not included."));
    }
};
/*
    This taskpane allows users to concatenate two or more
    dataframes together.
*/
const ConcatTaskpane = (props) => {
    var _a;
    const { params, setParams } = useLiveUpdatingParams({
        join: 'inner',
        ignore_index: true,
        sheet_indexes: []
    }, StepType.Concat, props.mitoAPI, props.analysisData, 50 // 50 ms debounce delay
    );
    // Make sure the user cannot select the newly created dataframe
    const [selectableSheetIndexes] = useState(props.sheetDataArray.map((sd, index) => index));
    // For each sheet concatonated together, find all of the columns that are not included in the final result
    const concatSheetColumnHeaders = Object.values(((_a = props.sheetDataArray[props.sheetDataArray.length - 1]) === null || _a === void 0 ? void 0 : _a.columnIDsMap) || {});
    const notIncludedColumnsArray = params === null || params === void 0 ? void 0 : params.sheet_indexes.map(sheetIndex => {
        var _a;
        return Object.values(((_a = props.sheetDataArray[sheetIndex]) === null || _a === void 0 ? void 0 : _a.columnIDsMap) || {}).filter(columnHeader => {
            // Because concat_edit makes a new sheet and you can't reopen the concat taskpane or reorder sheets,
            // the sheet this taskpane creates is always the last sheet in the sheetDataArray 
            return !concatSheetColumnHeaders.includes(columnHeader);
        });
    });
    if (params === undefined) {
        return (React.createElement(DefaultEmptyTaskpane, { setUIState: props.setUIState, message: "Import at least two datasets before concating." }));
    }
    const dataframeCards = params.sheet_indexes.map((sheetIndex, arrIndex) => {
        return (React.createElement("div", { key: arrIndex },
            React.createElement(SelectAndXIconCard, { titleMap: Object.fromEntries(props.sheetDataArray.map((sheetData, index) => {
                    return [index + '', sheetData.dfName];
                })), value: sheetIndex + '', onChange: (newSheetIndexStr) => {
                    const newSheetIndex = parseInt(newSheetIndexStr);
                    setParams(prevConcatParams => {
                        const newSheetIndexes = [...prevConcatParams.sheet_indexes];
                        newSheetIndexes[arrIndex] = newSheetIndex;
                        return Object.assign(Object.assign({}, prevConcatParams), { sheet_indexes: newSheetIndexes });
                    });
                }, onDelete: () => {
                    setParams(prevConcatParams => {
                        const newSheetIndexes = [...prevConcatParams.sheet_indexes];
                        newSheetIndexes.splice(arrIndex, 1);
                        return Object.assign(Object.assign({}, prevConcatParams), { sheet_indexes: newSheetIndexes });
                    });
                }, selectableValues: Object.keys(props.sheetDataArray) }),
            notIncludedColumnsArray !== undefined &&
                React.createElement(Row, { className: 'text-subtext-1' }, getColumnHeadersIncludedMessage(notIncludedColumnsArray, arrIndex))));
    });
    return (React.createElement(DefaultTaskpane, null,
        React.createElement(DefaultTaskpaneHeader, { header: "Concatenate Sheet", setUIState: props.setUIState }),
        React.createElement(DefaultTaskpaneBody, null,
            React.createElement(Row, { justify: 'space-between', align: 'center' },
                React.createElement(Col, null,
                    React.createElement("p", { className: 'text-header-3' }, "Join Type")),
                React.createElement(Col, null,
                    React.createElement(Select, { value: params.join, onChange: (newJoin) => {
                            setParams(prevConcatParams => {
                                return Object.assign(Object.assign({}, prevConcatParams), { join: newJoin });
                            });
                        }, width: 'medium' },
                        React.createElement(DropdownItem, { title: 'inner', subtext: "Only includes columns that exist in all sheets" }),
                        React.createElement(DropdownItem, { title: "outer", subtext: "Includes all columns from all sheets, regardless of if these columns are in the other sheets." })))),
            React.createElement(Row, { justify: 'space-between', align: 'center' },
                React.createElement(Col, null,
                    React.createElement(Row, { align: 'center', suppressTopBottomMargin: true },
                        React.createElement("p", { className: 'text-header-3' }, "Ignore Existing Indexes \u00A0"),
                        React.createElement(Tooltip, { title: "When on, the resulting dataframe will have indexes 0, 1, 2, etc.. This is useful if you're concatenating objects that don't have meaningful index information." }))),
                React.createElement(Col, null,
                    React.createElement(Toggle, { value: params.ignore_index, onChange: () => {
                            setParams(prevConcatParams => {
                                return Object.assign(Object.assign({}, prevConcatParams), { ignore_index: !prevConcatParams.ignore_index });
                            });
                        } }))),
            React.createElement(Row, { justify: 'space-between', align: 'center' },
                React.createElement(Col, null,
                    React.createElement("p", { className: 'text-header-3' }, "Dataframes to Concatenate")),
                React.createElement(Col, null,
                    React.createElement(DropdownButton, { text: '+ Add', width: 'small', searchable: true }, [
                        React.createElement(DropdownItem, { key: -1, title: "Add all sheets", onClick: () => {
                                setParams(prevConcatParams => {
                                    const newSheetIndexes = [...selectableSheetIndexes];
                                    return Object.assign(Object.assign({}, prevConcatParams), { sheet_indexes: newSheetIndexes });
                                });
                            } })
                    ].concat(props.sheetDataArray.filter((sheetData, index) => {
                        if (!selectableSheetIndexes.includes(index)) {
                            return false;
                        }
                        return true;
                    }).map((sheetData, index) => {
                        return (React.createElement(DropdownItem, { key: index, title: sheetData.dfName, onClick: () => {
                                setParams(prevConcatParams => {
                                    const newSheetIndexes = [...prevConcatParams.sheet_indexes];
                                    newSheetIndexes.push(index);
                                    return Object.assign(Object.assign({}, prevConcatParams), { sheet_indexes: newSheetIndexes });
                                });
                            } }));
                    }))))),
            dataframeCards)));
};
export default ConcatTaskpane;
//# sourceMappingURL=ConcatTaskpane.js.map