# -------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
All custom model containers are listed here.
"""

import numpy as np
from onnxconverter_common.container import CommonSklearnModelContainer
import torch

from .operator_converters import constants


class CommonONNXModelContainer(CommonSklearnModelContainer):
    """
    Common container for input ONNX operators.
    """

    def __init__(self, onnx_model):
        super(CommonONNXModelContainer, self).__init__(onnx_model)


class PyTorchBackendModel(torch.nn.Module):
    """
    Container for a model compiled into PyTorch.
    """

    def __init__(self, input_names, output_names, operator_map, operators, extra_config):
        """
        Args:
            input_names: The names of the input `onnxconverter_common.topology.Variable`s for this model
            output_names: The names of the output `onnxconverter_common.topology.Variable`s generated by this model
            operator_map: A dictionary of operator aliases and related PyTorch implementations
            operators: The list of operators (in a topological order) that will be executed by the model (in order)
            extra_config: Some additional custom configuration parameter
        """
        super(PyTorchBackendModel, self).__init__()
        self.input_names = input_names
        self.output_names = output_names
        self.operator_map = torch.nn.ModuleDict(operator_map)
        self.operators = operators
        self.extra_config = extra_config
        self.is_regression = self.operator_map[self.operators[-1].full_name].regression
        self.anomaly_detection = self.operator_map[self.operators[-1].full_name].anomaly_detection

    def forward(self, *inputs):
        with torch.no_grad():
            inputs = [*inputs]
            variable_map = {}
            device = None
            if len(list(self.parameters())) > 0:
                device = next(self.parameters()).device  # Assuming we are using a single device for all parameters

            # Maps data inputs to the expected variables.
            for i, input_name in enumerate(self.input_names):
                if type(inputs[i]) is np.ndarray:
                    inputs[i] = torch.from_numpy(inputs[i]).float()
                elif type(inputs[i]) is not torch.Tensor:
                    raise RuntimeError("Inputer tensor {} of not supported type {}".format(input_name, type(inputs[i])))
                if device is not None:
                    inputs[i] = inputs[i].to(device)
                variable_map[input_name] = inputs[i]

            # Evaluate all the operators in the topology by properly wiring inputs \ outputs
            for operator in self.operators:
                pytorch_op = self.operator_map[operator.full_name]
                pytorch_outputs = pytorch_op(*(variable_map[input] for input in operator.input_full_names))

                if len(operator.output_full_names) == 1:
                    variable_map[operator.output_full_names[0]] = pytorch_outputs
                else:
                    for i, output in enumerate(operator.output_full_names):
                        variable_map[output] = pytorch_outputs[i]

            # Prepare and return the output.
            if len(self.output_names) == 1:
                return variable_map[self.output_names[0]]
            else:
                return list(variable_map[output_name] for output_name in self.output_names)


class PyTorchBackendModelTransformer(PyTorchBackendModel):
    def transform(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On data transformers it returns transformed output data
        """
        return self.forward(*inputs).cpu().numpy()


class PyTorchBackendModelRegression(PyTorchBackendModel):
    def predict(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On regression returns the predicted values.
        On classification tasks returns the predicted class labels for the input data.
        On anomaly detection (e.g. isolation forest) returns the predicted classes (-1 or 1).
        """
        if self.is_regression:
            return self.forward(*inputs).cpu().numpy().flatten()
        elif self.anomaly_detection:
            return self.forward(*inputs)[0].cpu().numpy().flatten()
        else:
            return self.forward(*inputs)[0].cpu().numpy()


class PyTorchBackendModelClassification(PyTorchBackendModelRegression):
    def predict_proba(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On classification tasks returns the probability estimates.
        """
        return self.forward(*inputs)[1].cpu().numpy()


class PyTorchBackendModelAnomalyDetection(PyTorchBackendModelRegression):
    def decision_function(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On anomaly detection (e.g. isolation forest) returns the decision function scores.
        """
        return self.forward(*inputs)[1].cpu().numpy().flatten()

    def score_samples(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On anomaly detection (e.g. isolation forest) returns the decision_function score plus offset_
        """
        return self.decision_function(*inputs) + self.extra_config[constants.OFFSET]
