"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventsRuleToKinesisStreams = void 0;
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const core_2 = require("@aws-solutions-constructs/core");
class EventsRuleToKinesisStreams extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the EventsRuleToKinesisStreams class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Set up the Kinesis Stream
        this.kinesisStream = defaults.buildKinesisStream(this, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps,
        });
        // Create an events service role
        this.eventsRole = new iam.Role(this, 'eventsRole', {
            assumedBy: new iam.ServicePrincipal('events.amazonaws.com'),
            description: 'Events Rule Role',
        });
        // Grant permission to events service role to allow event rule to send events data to the kinesis stream
        this.kinesisStream.grantWrite(this.eventsRole);
        // Set up the Kinesis Stream as the target for event rule
        const kinesisStreamEventTarget = {
            bind: () => ({
                id: '',
                arn: this.kinesisStream.streamArn,
                role: this.eventsRole
            })
        };
        // Set up the events rule props
        const defaultEventsRuleProps = defaults.DefaultEventsRuleProps([kinesisStreamEventTarget]);
        const eventsRuleProps = core_2.overrideProps(defaultEventsRuleProps, props.eventRuleProps, true);
        // Setup up the event rule
        this.eventsRule = new events.Rule(this, 'EventsRule', eventsRuleProps);
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for Kinesis Stream
            this.cloudwatchAlarms = defaults.buildKinesisStreamCWAlarms(this);
        }
    }
}
exports.EventsRuleToKinesisStreams = EventsRuleToKinesisStreams;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7OztBQUVILDhDQUE4QztBQUU5Qyx3Q0FBd0M7QUFFeEMsd0NBQTBDO0FBQzFDLDJEQUEyRDtBQUMzRCx5REFBK0Q7QUFnQy9ELE1BQWEsMEJBQTJCLFNBQVEsZ0JBQVM7Ozs7Ozs7OztJQWNyRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNDO1FBQzlFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsNEJBQTRCO1FBQzVCLElBQUksQ0FBQyxhQUFhLEdBQUcsUUFBUSxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRTtZQUNyRCxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO1lBQzFDLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7U0FDN0MsQ0FBQyxDQUFDO1FBRUgsZ0NBQWdDO1FBQ2hDLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDakQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQzNELFdBQVcsRUFBRSxrQkFBa0I7U0FDaEMsQ0FBQyxDQUFDO1FBRUgsd0dBQXdHO1FBQ3hHLElBQUksQ0FBQyxhQUFhLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUUvQyx5REFBeUQ7UUFDekQsTUFBTSx3QkFBd0IsR0FBdUI7WUFDbkQsSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7Z0JBQ1gsRUFBRSxFQUFFLEVBQUU7Z0JBQ04sR0FBRyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUztnQkFDakMsSUFBSSxFQUFFLElBQUksQ0FBQyxVQUFVO2FBQ3RCLENBQUM7U0FDSCxDQUFDO1FBRUYsK0JBQStCO1FBQy9CLE1BQU0sc0JBQXNCLEdBQUcsUUFBUSxDQUFDLHNCQUFzQixDQUFDLENBQUMsd0JBQXdCLENBQUMsQ0FBQyxDQUFDO1FBQzNGLE1BQU0sZUFBZSxHQUFHLG9CQUFhLENBQUMsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLGNBQWMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUUxRiwwQkFBMEI7UUFDMUIsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRSxlQUFlLENBQUMsQ0FBQztRQUV2RSxJQUFJLEtBQUssQ0FBQyxzQkFBc0IsS0FBSyxTQUFTLElBQUksS0FBSyxDQUFDLHNCQUFzQixFQUFFO1lBQzlFLHFEQUFxRDtZQUNyRCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsUUFBUSxDQUFDLDBCQUEwQixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ25FO0lBQ0gsQ0FBQztDQUNKO0FBckRELGdFQXFEQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIxIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1ldmVudHMnO1xuaW1wb3J0ICogYXMga2luZXNpcyBmcm9tICdAYXdzLWNkay9hd3Mta2luZXNpcyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgZGVmYXVsdHMgZnJvbSAnQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlJztcbmltcG9ydCB7IG92ZXJyaWRlUHJvcHMgfSBmcm9tICdAYXdzLXNvbHV0aW9ucy1jb25zdHJ1Y3RzL2NvcmUnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEV2ZW50c1J1bGVUb0tpbmVzaXNTdHJlYW1zUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbnJlYWRvbmx5IGV2ZW50UnVsZVByb3BzOiBldmVudHMuUnVsZVByb3BzXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxucmVhZG9ubHkgZXhpc3RpbmdTdHJlYW1PYmo/OiBraW5lc2lzLlN0cmVhbSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbnJlYWRvbmx5IGtpbmVzaXNTdHJlYW1Qcm9wcz86IGtpbmVzaXMuU3RyZWFtUHJvcHMgfCBhbnlcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbnJlYWRvbmx5IGNyZWF0ZUNsb3VkV2F0Y2hBbGFybXM/OiBib29sZWFuXG59XG5cbmV4cG9ydCBjbGFzcyBFdmVudHNSdWxlVG9LaW5lc2lzU3RyZWFtcyBleHRlbmRzIENvbnN0cnVjdCB7XG4gICAgcHVibGljIHJlYWRvbmx5IGtpbmVzaXNTdHJlYW06IGtpbmVzaXMuU3RyZWFtO1xuICAgIHB1YmxpYyByZWFkb25seSBldmVudHNSdWxlOiBldmVudHMuUnVsZTtcbiAgICBwdWJsaWMgcmVhZG9ubHkgZXZlbnRzUm9sZTogaWFtLlJvbGU7XG4gICAgcHVibGljIHJlYWRvbmx5IGNsb3Vkd2F0Y2hBbGFybXM/OiBjbG91ZHdhdGNoLkFsYXJtW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRXZlbnRzUnVsZVRvS2luZXNpc1N0cmVhbXNQcm9wcykge1xuICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgICAgLy8gU2V0IHVwIHRoZSBLaW5lc2lzIFN0cmVhbVxuICAgICAgdGhpcy5raW5lc2lzU3RyZWFtID0gZGVmYXVsdHMuYnVpbGRLaW5lc2lzU3RyZWFtKHRoaXMsIHtcbiAgICAgICAgZXhpc3RpbmdTdHJlYW1PYmo6IHByb3BzLmV4aXN0aW5nU3RyZWFtT2JqLFxuICAgICAgICBraW5lc2lzU3RyZWFtUHJvcHM6IHByb3BzLmtpbmVzaXNTdHJlYW1Qcm9wcyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBDcmVhdGUgYW4gZXZlbnRzIHNlcnZpY2Ugcm9sZVxuICAgICAgdGhpcy5ldmVudHNSb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdldmVudHNSb2xlJywge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZXZlbnRzLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgZGVzY3JpcHRpb246ICdFdmVudHMgUnVsZSBSb2xlJyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBHcmFudCBwZXJtaXNzaW9uIHRvIGV2ZW50cyBzZXJ2aWNlIHJvbGUgdG8gYWxsb3cgZXZlbnQgcnVsZSB0byBzZW5kIGV2ZW50cyBkYXRhIHRvIHRoZSBraW5lc2lzIHN0cmVhbVxuICAgICAgdGhpcy5raW5lc2lzU3RyZWFtLmdyYW50V3JpdGUodGhpcy5ldmVudHNSb2xlKTtcblxuICAgICAgLy8gU2V0IHVwIHRoZSBLaW5lc2lzIFN0cmVhbSBhcyB0aGUgdGFyZ2V0IGZvciBldmVudCBydWxlXG4gICAgICBjb25zdCBraW5lc2lzU3RyZWFtRXZlbnRUYXJnZXQ6IGV2ZW50cy5JUnVsZVRhcmdldCA9IHtcbiAgICAgICAgYmluZDogKCkgPT4gKHtcbiAgICAgICAgICBpZDogJycsXG4gICAgICAgICAgYXJuOiB0aGlzLmtpbmVzaXNTdHJlYW0uc3RyZWFtQXJuLFxuICAgICAgICAgIHJvbGU6IHRoaXMuZXZlbnRzUm9sZVxuICAgICAgICB9KVxuICAgICAgfTtcblxuICAgICAgLy8gU2V0IHVwIHRoZSBldmVudHMgcnVsZSBwcm9wc1xuICAgICAgY29uc3QgZGVmYXVsdEV2ZW50c1J1bGVQcm9wcyA9IGRlZmF1bHRzLkRlZmF1bHRFdmVudHNSdWxlUHJvcHMoW2tpbmVzaXNTdHJlYW1FdmVudFRhcmdldF0pO1xuICAgICAgY29uc3QgZXZlbnRzUnVsZVByb3BzID0gb3ZlcnJpZGVQcm9wcyhkZWZhdWx0RXZlbnRzUnVsZVByb3BzLCBwcm9wcy5ldmVudFJ1bGVQcm9wcywgdHJ1ZSk7XG5cbiAgICAgIC8vIFNldHVwIHVwIHRoZSBldmVudCBydWxlXG4gICAgICB0aGlzLmV2ZW50c1J1bGUgPSBuZXcgZXZlbnRzLlJ1bGUodGhpcywgJ0V2ZW50c1J1bGUnLCBldmVudHNSdWxlUHJvcHMpO1xuXG4gICAgICBpZiAocHJvcHMuY3JlYXRlQ2xvdWRXYXRjaEFsYXJtcyA9PT0gdW5kZWZpbmVkIHx8IHByb3BzLmNyZWF0ZUNsb3VkV2F0Y2hBbGFybXMpIHtcbiAgICAgICAgLy8gRGVwbG95IGJlc3QgcHJhY3RpY2VzIENXIEFsYXJtcyBmb3IgS2luZXNpcyBTdHJlYW1cbiAgICAgICAgdGhpcy5jbG91ZHdhdGNoQWxhcm1zID0gZGVmYXVsdHMuYnVpbGRLaW5lc2lzU3RyZWFtQ1dBbGFybXModGhpcyk7XG4gICAgICB9XG4gICAgfVxufSJdfQ==