"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTrainingJob = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
const utils_1 = require("./private/utils");
/**
 * (experimental) Class representing the SageMaker Create Training Job task.
 *
 * @experimental
 */
class SageMakerCreateTrainingJob extends sfn.TaskStateBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * (experimental) Allows specify security group connections for instances of this fleet.
         *
         * @experimental
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSize: core_1.Size.gibibytes(10),
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1),
        };
        // check that either algorithm name or image is defined
        if (!props.algorithmSpecification.algorithmName && !props.algorithmSpecification.trainingImage) {
            throw new Error('Must define either an algorithm name or training image URI in the algorithm specification');
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = props.algorithmSpecification.trainingInputMode
            ? props.algorithmSpecification
            : { ...props.algorithmSpecification, trainingInputMode: base_types_1.InputMode.FILE };
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map((config) => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return {
                    ...config,
                    dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
                };
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = props.vpcConfig.subnets ? this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * (experimental) The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     *
     * @experimental
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    /**
     * (experimental) The principal to grant permissions to.
     *
     * @experimental
     */
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error('Principal not available yet--use the object in a Task first');
        }
        return this._grantPrincipal;
    }
    /**
     * (experimental) Add the security group to all instances via the launch configuration security groups array.
     *
     * @param securityGroup : The security group to add.
     * @experimental
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTrainingJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            RoleArn: this._role.roleArn,
            ...this.renderAlgorithmSpecification(this.algorithmSpecification),
            ...this.renderInputDataConfig(this.inputDataConfig),
            ...this.renderOutputDataConfig(this.props.outputDataConfig),
            ...this.renderResourceConfig(this.resourceConfig),
            ...this.renderStoppingCondition(this.stoppingCondition),
            ...this.renderHyperparameters(this.props.hyperparameters),
            ...utils_1.renderTags(this.props.tags),
            ...this.renderVpcConfig(this.props.vpcConfig),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {}),
                ...(spec.algorithmName ? { AlgorithmName: spec.algorithmName } : {}),
                ...(spec.metricDefinitions
                    ? { MetricDefinitions: spec.metricDefinitions.map((metric) => ({ Name: metric.name, Regex: metric.regex })) }
                    : {}),
            },
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map((channel) => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType
                            ? { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType }
                            : {}),
                        ...(channel.dataSource.s3DataSource.attributeNames ? { AttributeNames: channel.dataSource.s3DataSource.attributeNames } : {}),
                    },
                },
                ...(channel.compressionType ? { CompressionType: channel.compressionType } : {}),
                ...(channel.contentType ? { ContentType: channel.contentType } : {}),
                ...(channel.inputMode ? { InputMode: channel.inputMode } : {}),
                ...(channel.recordWrapperType ? { RecordWrapperType: channel.recordWrapperType } : {}),
            })),
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey ? { KmsKeyId: config.encryptionKey.keyArn } : {}),
            },
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: 'ml.' + config.instanceType,
                VolumeSizeInGB: config.volumeSize.toGibibytes(),
                ...(config.volumeEncryptionKey ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds(),
            },
        };
    }
    renderHyperparameters(params) {
        return params ? { HyperParameters: params } : {};
    }
    renderVpcConfig(config) {
        return config
            ? {
                VpcConfig: {
                    SecurityGroupIds: core_1.Lazy.list({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
    makePolicyStatements() {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role =
            this.props.role ||
                new iam.Role(this, 'SagemakerRole', {
                    assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                    inlinePolicies: {
                        CreateTrainingJob: new iam.PolicyDocument({
                            statements: [
                                new iam.PolicyStatement({
                                    actions: [
                                        'cloudwatch:PutMetricData',
                                        'logs:CreateLogStream',
                                        'logs:PutLogEvents',
                                        'logs:CreateLogGroup',
                                        'logs:DescribeLogStreams',
                                        'ecr:GetAuthorizationToken',
                                        ...(this.props.vpcConfig
                                            ? [
                                                'ec2:CreateNetworkInterface',
                                                'ec2:CreateNetworkInterfacePermission',
                                                'ec2:DeleteNetworkInterface',
                                                'ec2:DeleteNetworkInterfacePermission',
                                                'ec2:DescribeNetworkInterfaces',
                                                'ec2:DescribeVpcs',
                                                'ec2:DescribeDhcpOptions',
                                                'ec2:DescribeSubnets',
                                                'ec2:DescribeSecurityGroups',
                                            ]
                                            : []),
                                    ],
                                    resources: ['*'],
                                }),
                            ],
                        }),
                    },
                });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'TrainJobSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        const stack = core_1.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTrainingJob = SageMakerCreateTrainingJob;
SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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