"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateModel = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * (experimental) A Step Functions Task to create a SageMaker model.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html
 * @experimental
 */
class SageMakerCreateModel extends sfn.TaskStateBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * (experimental) Allows specify security group connections for instances of this fleet.
         *
         * @experimental
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateModel.SUPPORTED_INTEGRATION_PATTERNS);
        // add the security groups to the connections object
        if (props.vpc) {
            this.vpc = props.vpc;
            this.subnets = props.subnetSelection ? this.vpc.selectSubnets(props.subnetSelection).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.role = this.props.role || this.createSagemakerRole();
        this.grantPrincipal = this.role;
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * (experimental) Add the security group to all instances via the launch configuration security groups array.
     *
     * @param securityGroup : The security group to add.
     * @experimental
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createModel', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        var _a, _b;
        return {
            EnableNetworkIsolation: this.props.enableNetworkIsolation,
            ExecutionRoleArn: this.role.roleArn,
            ModelName: this.props.modelName,
            Tags: (_a = this.props.tags) === null || _a === void 0 ? void 0 : _a.value,
            PrimaryContainer: this.props.primaryContainer.bind(this).parameters,
            Containers: (_b = this.props.containers) === null || _b === void 0 ? void 0 : _b.map(container => (container.bind(this))),
            ...this.renderVpcConfig(),
        };
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        return [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateModel'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'model',
                        // If the model name comes from input, we cannot target the policy to a particular ARN prefix reliably.
                        // SageMaker uses lowercase for resource name in the arn
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.modelName) ? '*' : `${this.props.modelName.toLowerCase()}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
    }
    createSagemakerRole() {
        // https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-roles.html
        const role = new iam.Role(this, 'SagemakerRole', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            inlinePolicies: {
                CreateModel: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatch.html
                                'cloudwatch:PutMetricData',
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchlogs.html
                                'logs:CreateLogStream',
                                'logs:CreateLogGroup',
                                'logs:PutLogEvents',
                                'logs:DescribeLogStreams',
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerregistry.html
                                'ecr:GetAuthorizationToken',
                            ],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        if (this.props.vpc) {
            role.addToPrincipalPolicy(
            // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            new iam.PolicyStatement({
                actions: [
                    'ec2:CreateNetworkInterface',
                    'ec2:CreateNetworkInterfacePermission',
                    'ec2:DeleteNetworkInterface',
                    'ec2:DeleteNetworkInterfacePermission',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DescribeVpcs',
                    'ec2:DescribeDhcpOptions',
                    'ec2:DescribeSubnets',
                    'ec2:DescribeSecurityGroups',
                ],
                resources: ['*'],
            }));
        }
        return role;
    }
    renderVpcConfig() {
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'ModelSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        return this.vpc
            ? {
                VpcConfig: {
                    SecurityGroupIds: cdk.Lazy.list({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
}
exports.SageMakerCreateModel = SageMakerCreateModel;
SageMakerCreateModel.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
];
//# sourceMappingURL=data:application/json;base64,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