#!/usr/bin/env python

#  Copyright (c) 2019 JD Williams
#
#  This file is part of Firefly, a Python SOA framework built by JD Williams. Firefly is free software; you can
#  redistribute it and/or modify it under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 3 of the License, or (at your option) any later version.
#
#  Firefly is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
#  Public License for more details. You should have received a copy of the GNU Lesser General Public
#  License along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#  You should have received a copy of the GNU General Public License along with Firefly. If not, see
#  <http://www.gnu.org/licenses/>.

"""Unit tests for the input.py file."""

import gyp.input
import unittest
import sys


class TestFindCycles(unittest.TestCase):
  def setUp(self):
    self.nodes = {}
    for x in ('a', 'b', 'c', 'd', 'e'):
      self.nodes[x] = gyp.input.DependencyGraphNode(x)

  def _create_dependency(self, dependent, dependency):
    dependent.dependencies.append(dependency)
    dependency.dependents.append(dependent)

  def test_no_cycle_empty_graph(self):
    for label, node in self.nodes.iteritems():
      self.assertEquals([], node.FindCycles())

  def test_no_cycle_line(self):
    self._create_dependency(self.nodes['a'], self.nodes['b'])
    self._create_dependency(self.nodes['b'], self.nodes['c'])
    self._create_dependency(self.nodes['c'], self.nodes['d'])

    for label, node in self.nodes.iteritems():
      self.assertEquals([], node.FindCycles())

  def test_no_cycle_dag(self):
    self._create_dependency(self.nodes['a'], self.nodes['b'])
    self._create_dependency(self.nodes['a'], self.nodes['c'])
    self._create_dependency(self.nodes['b'], self.nodes['c'])

    for label, node in self.nodes.iteritems():
      self.assertEquals([], node.FindCycles())

  def test_cycle_self_reference(self):
    self._create_dependency(self.nodes['a'], self.nodes['a'])

    self.assertEquals([[self.nodes['a'], self.nodes['a']]],
                      self.nodes['a'].FindCycles())

  def test_cycle_two_nodes(self):
    self._create_dependency(self.nodes['a'], self.nodes['b'])
    self._create_dependency(self.nodes['b'], self.nodes['a'])

    self.assertEquals([[self.nodes['a'], self.nodes['b'], self.nodes['a']]],
                      self.nodes['a'].FindCycles())
    self.assertEquals([[self.nodes['b'], self.nodes['a'], self.nodes['b']]],
                      self.nodes['b'].FindCycles())

  def test_two_cycles(self):
    self._create_dependency(self.nodes['a'], self.nodes['b'])
    self._create_dependency(self.nodes['b'], self.nodes['a'])

    self._create_dependency(self.nodes['b'], self.nodes['c'])
    self._create_dependency(self.nodes['c'], self.nodes['b'])

    cycles = self.nodes['a'].FindCycles()
    self.assertTrue(
       [self.nodes['a'], self.nodes['b'], self.nodes['a']] in cycles)
    self.assertTrue(
       [self.nodes['b'], self.nodes['c'], self.nodes['b']] in cycles)
    self.assertEquals(2, len(cycles))

  def test_big_cycle(self):
    self._create_dependency(self.nodes['a'], self.nodes['b'])
    self._create_dependency(self.nodes['b'], self.nodes['c'])
    self._create_dependency(self.nodes['c'], self.nodes['d'])
    self._create_dependency(self.nodes['d'], self.nodes['e'])
    self._create_dependency(self.nodes['e'], self.nodes['a'])

    self.assertEquals([[self.nodes['a'],
                        self.nodes['b'],
                        self.nodes['c'],
                        self.nodes['d'],
                        self.nodes['e'],
                        self.nodes['a']]],
                      self.nodes['a'].FindCycles())


if __name__ == '__main__':
  unittest.main()
