"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmtpCredentials = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const smtp_credentials_provider_1 = require("./smtp-credentials-provider");
/**
 * This construct creates an IAM user, with a policy permitting emails to be sent via SES from
 * a specified email address, creates an access key associated with this user, and converts the
 * access key to SMTP credentials.
 *
 * The generated SMTP credentials are stored as a parameter in Parameter Store, and the name of
 * this parameter is output as a CloudFormation output. The parameter may be safely deleted, once
 * the credentials have been accessed.
 *
 * @example
 *
 * new SmtpCredentials(this, 'SmtpCredentials', {
 *     emailAddress: 'me@charles.fish',
 * });
 */
class SmtpCredentials extends constructs_1.Construct {
    /**
     * @param scope A reference to the stack which this construct will be created in. Note that the
     * SMTP credentials generated will only be permitted to send emails in this stack's region.
     *
     * @param id A unique identifier, within the context that this construct is created.
     *
     * @param props Configuration defining how this construct should be created.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { emailAddress } = props;
        const domainName = this.extractDomainName(emailAddress);
        const { userArn, userName } = this.createIamUser(domainName, emailAddress);
        const { accessKey, smtpPassword } = this.createSmtpCredentials(userArn, userName);
        new aws_cdk_lib_1.CfnOutput(this, "SmtpCredentialsParameterName", {
            value: new aws_cdk_lib_1.aws_ssm.StringParameter(this, "SmtpCredentials", {
                stringValue: JSON.stringify({
                    AccessKey: accessKey,
                    SmtpPassword: smtpPassword,
                }),
            }).parameterName,
        });
    }
    extractDomainName(emailAddress) {
        if (emailAddress.indexOf("@") === -1) {
            throw Error("Invalid email address supplied.");
        }
        return emailAddress.substring(emailAddress.lastIndexOf("@") + 1);
    }
    createIamUser(domainName, emailAddress) {
        const user = new aws_cdk_lib_1.aws_iam.User(this, "SmtpUser");
        user.attachInlinePolicy(new aws_cdk_lib_1.aws_iam.Policy(this, "SmtpPolicy", {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                    actions: ["ses:SendRawEmail"],
                    resources: [
                        `arn:aws:ses:${process.env.CDK_DEFAULT_REGION}:${process.env.CDK_DEFAULT_ACCOUNT}:identity/${domainName}`,
                    ],
                    conditions: {
                        StringEquals: {
                            "ses:FromAddress": emailAddress,
                        },
                    },
                }),
            ],
        }));
        return user;
    }
    createSmtpCredentials(userArn, userName) {
        const { serviceToken } = new smtp_credentials_provider_1.SmtpCredentialsProvider(this, "SmtpCredentialsProvider", { userArn });
        const credentials = new aws_cdk_lib_1.CustomResource(this, "SmtpCredentialsLambda", {
            serviceToken,
            properties: {
                UserName: userName,
            },
        });
        return {
            accessKey: credentials.getAttString("AccessKey"),
            smtpPassword: credentials.getAttString("SmtpPassword"),
        };
    }
}
exports.SmtpCredentials = SmtpCredentials;
_a = JSII_RTTI_SYMBOL_1;
SmtpCredentials[_a] = { fqn: "@charlesdotfish/smtp-credentials-cdk-construct.SmtpCredentials", version: "2.0.0" };
//# sourceMappingURL=data:application/json;base64,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