import clr
import os
from clicknium.common.constants import _Constants, _ExceptionNames
from clicknium.common.enums import *
from clicknium.common.enums import _InternalClickType
from clicknium.common.models.exceptions import *
from clicknium.common.utils import Utils
from clicknium.common.models.locatoritem import LocatorItem

source_path = Utils.get_automation_libfolder()
black_dlls = ['Clicknium.Desktop.IA.Core.dll']
dlls = Utils.get_import_dlls(source_path)
for dll in dlls:
    if dll not in black_dlls:
        dll_path = os.path.join(source_path, dll)
        clr.AddReference(dll_path)

from Clicknium.Wrapper import UiElementRunner, BrowserTabExtension
from Clicknium.Wrapper.Options import *
from Clicknium.Common import BrowserType as dotnet_BrowserType, AutomationTech as dotnet_AutomationTech
from Clicknium.Common.Exception import ElementNotFoundException, DesktopException, WebException
from Clicknium.Common.Options import BringWindowType as dotnet_BringWindowType, InputMethod as dotnet_InputMethod, ClickType as dotnet_ClickType, \
                                    MouseButton as dotnet_MouseButton, ClickLocation as dotnet_ClickLocation, ClickMethod as dotnet_ClickMethod, \
                                    ModifierKeysEnum as dotnet_ModifierKeysEnum, PrefixAction as dotnet_PrefixAction, CheckType as dotnet_CheckType, \
                                    WebUserDataFolderMode as dotnet_WebUserDataFolderMode, OpenBrowserOption as dotnet_OpenBrowserOption, \
                                    ClearMethod as dotnet_ClearMethod, ClearHotKey as dotnet_ClearHotKey
from Clicknium.LocatorStore import LocatorStoreManager
from Clicknium.LocatorStore.Common.Exceptions import UnAuthorizedException
from System.Collections.Generic import Dictionary
from System import *
from System.Drawing import ColorTranslator
from Clicknium.Telemetry import TelemetryWrapper
from Clicknium.Telemetry.Models import EventData, EventTypes as dotnet_EventTypes

TelemetryInstance =  TelemetryWrapper("Python", Utils.PythonVersion, _Constants.SdkVersion)

class _ExceptionHandle:
    
    @staticmethod
    def try_except(func):
        def wrapper(*args, **kwargs):
            exception : Exception = None     
            properties = {_Constants.EventDataMethodKey: func.__name__}
            try:                  
                TelemetryService.report_event(EventTypes.ApiCall, properties)
                return func(*args, **kwargs)

            except ArgumentException as e:
                exception_name = e.__class__.__name__
                exception_message = e.Message
                exception_stacktrace = e.StackTrace
                if(exception_name.__eq__(_ExceptionNames.ArgumentNullException)):
                    exception = ArgumentNullError(exception_message, exception_stacktrace)
                elif(exception_name.__eq__(_ExceptionNames.ArgumentNullException)):
                    exception = ArgumentOutOfRangeError(exception_message, exception_stacktrace)
                else:
                    exception = ArgumentError(exception_message, exception_stacktrace)

            except NotSupportedException as e:
                exception_name = e.__class__.__name__
                exception_message = e.Message
                exception_stacktrace = e.StackTrace
                if(exception_name.__eq__(_ExceptionNames.OperationNotSupportedException)):
                    automation_tech = _ConvertDotnetEnumService.convert_automatiom_tech_method(e.AutoTech)
                    control_type = e.ControlType
                    operation = e.Operation
                    exception = NotSupportedOperationError(automation_tech, control_type, operation, exception_message, exception_stacktrace)
                elif(exception_name.__eq__(_ExceptionNames.OperationOptionNotSupportedException)):
                    automation_tech = _ConvertDotnetEnumService.convert_automatiom_tech_method(e.AutoTech)
                    control_type = e.ControlType
                    operation = e.Operation
                    option = e.Option
                    exception = NotSupportedOperationOptionError(automation_tech, control_type, operation, option, exception_message, exception_stacktrace)
                else:
                    exception = NotSupportedError(exception_message, exception_stacktrace)

            except InvalidOperationException as e:
                exception_name = e.__class__.__name__
                exception_message = e.Message
                exception_stacktrace = e.StackTrace
                if(exception_name.__eq__(_ExceptionNames.InvalidSelectedItemException)):
                    exception = InvalidSelectedItemError(e.Item, exception_message, exception_stacktrace)
                else: 
                    exception = InvalidOperationError(exception_message, exception_stacktrace)

            except TimeoutException as e:
                exception_message = e.Message
                exception_stacktrace = e.StackTrace
                exception = OperationTimeoutError(exception_message, exception_stacktrace)

            except ElementNotFoundException as e:
                exception_message = e.Message
                exception_stacktrace = e.StackTrace
                exception = ElementNotFoundError(exception_message, exception_stacktrace)

            except DesktopException as e:
                exception_name = e.__class__.__name__
                exception_message = e.Message
                exception_stacktrace = e.StackTrace
                if(exception_name.__eq__(_ExceptionNames.WindowsNativeException)):
                    name = e.Name
                    native_errorcode = e.NativeErrorCode
                    exception = WindowsNativeError(name, native_errorcode, exception_message, exception_stacktrace)
                elif(exception_name.__eq__(_ExceptionNames.UiaPatternNotFoundException)):
                    exception = UiaPatternNotFoundError(exception_message, exception_stacktrace)
                elif(exception_name.__eq__(_ExceptionNames.ScreenOperationException)):
                    name = e.Name
                    native_errorcode = e.NativeErrorCode
                    exception = ScreenOperationError(name, native_errorcode, exception_message, exception_stacktrace)
                else:
                    exception = WindowError(exception_message, exception_stacktrace)

            except WebException as e:
                exception_name = e.__class__.__name__
                exception_message = e.Message
                exception_stacktrace = e.StackTrace
                browser_type = _ConvertDotnetEnumService.convert_browser_type_method(e.BrowserType)                
                if(exception_name.__eq__(_ExceptionNames.BrowserNotRunException)):
                    exception = BrowserNotRunError(browser_type, exception_message, exception_stacktrace)
                elif(exception_name.__eq__(_ExceptionNames.BrowserNotInstallException)):
                    exception = BrowserNotInstallError(browser_type, exception_message, exception_stacktrace)
                elif(exception_name.__eq__(_ExceptionNames.UnreachableBrowserExtensionException)):
                    exception = UnreachableBrowserExtensionError(browser_type, exception_message, exception_stacktrace)
                elif(exception_name.__eq__(_ExceptionNames.WebElementNotRespondingException)):
                    exception = WebElementNotRespondingError(browser_type, exception_message, exception_stacktrace)  
                else:
                    exception = WebError(browser_type, exception_message, exception_stacktrace)

            except UnAuthorizedException as e:
                exception_message = e.Message
                exception_stacktrace = e.StackTrace
                exception = UnAuthorizedError(exception_message, exception_stacktrace)

            except BaseError as e:
                exception = e

            except Exception as e:
                exception = BaseError(e.Message, e.StackTrace)             

            if exception:
                # TrackException
                properties[_Constants.EventDataMessageKey] = exception.message
                properties[_Constants.EventDataStacktraceKey] = exception.stacktrace
                TelemetryService.report_event(EventTypes.ExceptionReport, properties)
                raise exception;        

        return wrapper


class _InvokerService:    

    # region BrowserTabExtension

    @staticmethod
    @_ExceptionHandle.try_except
    def find_element_web(tab, locator, locator_variables = {}):  
        locator_item = LocatorService.get_locator(locator, locator_variables)
        return BrowserTabExtension.GetElement(tab, locator_item.Locator, locator_item.Locator_Variables, locator_item.Locator_Image)

    @staticmethod
    @_ExceptionHandle.try_except
    def find_elements_web(tab, locator, locator_variables = {}, timeout = 30):  
        locator_item = LocatorService.get_locator(locator, locator_variables)
        return BrowserTabExtension.FindElements(tab, locator_item.Locator, locator_item.Locator_Variables, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def wait_disappear_web(tab, locator, locator_variables = {}, timeout = 30):   
        locator_item = LocatorService.get_locator(locator, locator_variables)   
        return BrowserTabExtension.WaitElementDisappear(tab, locator_item.Locator, locator_item.Locator_Variables, locator_item.Locator_Image, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def wait_appear_web(tab, locator, locator_variables = {}, timeout = 30):    
        locator_item = LocatorService.get_locator(locator, locator_variables)  
        return BrowserTabExtension.WaitElementAppear(tab, locator_item.Locator, locator_item.Locator_Variables, locator_item.Locator_Image, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def is_exist_web(tab, locator, locator_variables = {}, timeout = 30):
        locator_item = LocatorService.get_locator(locator, locator_variables)
        return BrowserTabExtension.ElementIsExist(tab, locator_item.Locator, locator_item.Locator_Variables, locator_item.Locator_Image, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def wait_property_web(tab, locator, name, value, locator_variables = {}, timeout = 30):
        locator_item = LocatorService.get_locator(locator, locator_variables)
        return BrowserTabExtension.WaitProperty(tab, locator_item.Locator, name, value, locator_item.Locator_Variables, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def set_property_web(tab, locator, name, value, locator_variables = {}, timeout = 30):
        locator_item = LocatorService.get_locator(locator, locator_variables)
        return BrowserTabExtension.SetProperty(tab, locator_item.Locator, name, value, locator_item.Locator_Variables, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def execute_js(tab, locator, javascript_code, method_invoke, locator_variables = {}, timeout = 30):
        locator_item = LocatorService.get_locator(locator, locator_variables)
        return BrowserTabExtension.ExecuteJavaScript(tab, locator_item.Locator, javascript_code, method_invoke, locator_item.Locator_Variables, timeout * 1000)

    # endregion BrowserTabExtension

    # region UiElementRunner

    @staticmethod
    @_ExceptionHandle.try_except
    def find_element(locator, locator_variables = {}, window_mode = WindowMode.Auto):       
        mode = _ConvertEnumService.convert_window_mode_enum(WindowMode(window_mode))    
        locator_item = LocatorService.get_locator(locator, locator_variables)          
        return UiElementRunner.GetElement(locator_item.Locator, locator_item.Locator_Variables, locator_item.Locator_Image, mode)

    @staticmethod
    @_ExceptionHandle.try_except
    def find_elements(locator, locator_variables = {}, timeout = 30):       
        locator_item = LocatorService.get_locator(locator, locator_variables)          
        return UiElementRunner.FindElements(locator_item.Locator, locator_item.Locator_Variables, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def get_webdriver(browser_type):
        return UiElementRunner.GetWebDriver(browser_type)

    @staticmethod
    @_ExceptionHandle.try_except
    def get_windowdriver():
        return UiElementRunner.GetDesktopDriver()

    @staticmethod
    @_ExceptionHandle.try_except
    def get_sapdriver():
        return UiElementRunner.GetSapDriver()

    @staticmethod
    @_ExceptionHandle.try_except
    def send_hotkey(hotkey):
        return UiElementRunner.SendHotKey(hotkey)

    @staticmethod
    @_ExceptionHandle.try_except
    def send_text(text):
        return UiElementRunner.InputText(text)

    @staticmethod
    @_ExceptionHandle.try_except
    def wait_disappear(locator, locator_variables = {}, timeout = 30):   
        locator_item = LocatorService.get_locator(locator, locator_variables)   
        return UiElementRunner.WaitElementDisappear(locator_item.Locator, locator_item.Locator_Variables, locator_item.Locator_Image, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def wait_appear(locator, locator_variables = {}, timeout = 30):    
        locator_item = LocatorService.get_locator(locator, locator_variables)  
        return UiElementRunner.WaitElementAppear(locator_item.Locator, locator_item.Locator_Variables, locator_item.Locator_Image, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def is_exist(locator, locator_variables = {}, timeout = 30):
        locator_item = LocatorService.get_locator(locator, locator_variables)
        return UiElementRunner.ElementIsExist(locator_item.Locator, locator_item.Locator_Variables, locator_item.Locator_Image, timeout * 1000)

    @staticmethod
    @_ExceptionHandle.try_except
    def wait_property(locator, name, value, locator_variables = {}, timeout = 30):
        locator_item = LocatorService.get_locator(locator, locator_variables)
        return UiElementRunner.WaitProperty(locator_item.Locator, name, value, locator_item.Locator_Variables, timeout * 1000)

    # endregion UiElementRunner


class LocatorService:

    @staticmethod
    def get_locator(locator, locator_variables = {}):
        project_folder = Utils.get_project_folder(_Constants.AppYaml)
        if not project_folder:
            project_folder = Utils.get_project_folder(_Constants.LocatorFolder)
        if not project_folder:
            raise ProjectSettingNotFoundError(_Constants.ProjectSettingNotFound)
        variables = _ConvertBaseTypeService.convert_dict(locator_variables) 
        locator_item = LocatorStoreManager.GetLocator(project_folder, str(locator), variables)
        if not locator_item:
            raise LocatorUndefinedError(_Constants.LocatorNotExist % locator)
        
        return LocatorItem(locator_item.Content, locator_item.Image, variables)


class TelemetryService:
    
    @staticmethod
    def report_event(type: str, properties: dict):
        data = _ConvertOptionService.convert_event_data(type, properties)
        TelemetryInstance.ReportEvent(data)


class _ConvertBaseTypeService:

    @staticmethod
    def convert_array(source_array):
        target_array = map(str, source_array)
        return Array[String](target_array)

    @staticmethod
    def convert_dict(source_dic):
        target_dict = Dictionary[String, Object]()
        for key in source_dic:
            target_dict[key] = source_dic[key]
        return target_dict

    @staticmethod
    def convert_dict_str(source_dic):
        target_dict = Dictionary[String, String]()
        for key in source_dic:
            target_dict[key] = source_dic[key]
        return target_dict

    @staticmethod
    def convert_color(color):
        return ColorTranslator.FromHtml(color)


class _ConvertOptionService:

    @staticmethod
    def convert_clickoption(click_type = _InternalClickType.Click, mouse_button = MouseButton.Left, click_location = ClickLocation.Center, 
                    click_method = ClickMethod.Default, modifier_key = ModifierKey.NoneKey, 
                    xoffset = 0, yoffset= 0, xrate=0, yrate = 0):
        click_option = ClickOption()        
        click_option.ClickType = _ConvertEnumService.convert_click_type_enum(click_type)
        click_option.MouseButton = _ConvertEnumService.convert_mouse_button_enum(MouseButton(mouse_button))
        click_option.ClickLocation = _ConvertEnumService.convert_click_location_enum(ClickLocation(click_location))
        click_option.ClickMethod = _ConvertEnumService.convert_click_method_enum(ClickMethod(click_method))
        click_option.ModifierKey = _ConvertEnumService.convert_modifier_key_enum(ModifierKey(modifier_key))
        click_option.XOffset = xoffset
        click_option.YOffset = yoffset
        click_option.XRate = xrate
        click_option.yrate = yrate
        return click_option

    @staticmethod
    def convert_settext_option(input_method = InputMethod.ControlSetValue):
        method = _ConvertEnumService.convert_input_method(InputMethod(input_method))
        settext_option = SetTextOption()
        settext_option.InputMethod = method
        return settext_option

    @staticmethod
    def convert_cleartext_option(clear_method, clear_hotkey= ClearHotKey.CtrlA_Delete, preaction = PreAction.SetFocus):
        clear_hotkey_option = ClearHotKeyOption()
        clear_hotkey_option.HotKey = _ConvertEnumService.convert_clear_hotkey_method(ClearHotKey(clear_hotkey))
        clear_hotkey_option.PrefixAction = _ConvertEnumService.convert_preaction_method(PreAction(preaction))
        cleartext_option = ClearTextOption()
        cleartext_option.ClearMethod = _ConvertEnumService.convert_clear_method(ClearMethod(clear_method))
        cleartext_option.ClearHotKeyOption = clear_hotkey_option
        return cleartext_option

    @staticmethod
    def convert_sendhotkey_option(preaction = PreAction.SetFocus):
        action = _ConvertEnumService.convert_preaction_method(PreAction(preaction))
        sendhotkey_option = SendHotKeyOption()
        sendhotkey_option.PrefixAction = action
        return sendhotkey_option

    @staticmethod
    def convert_dragdrop_option(xpoint = 0, ypoint = 0, speed = 50):
        dragdrop_option = DragdropOption()
        dragdrop_option.IntervalTime = speed
        dragdrop_option.XPoint = xpoint
        dragdrop_option.YPoint = ypoint
        return dragdrop_option

    @staticmethod
    def convert_check_option(check_type):
        check_option = CheckOption()
        check_option.CheckType = _ConvertEnumService.convert_checktype_method(CheckType(check_type))
        return check_option

    @staticmethod
    def convert_select_items_option(clear_selected):
        select_items_option = SelectMultipleItemOption()
        select_items_option.ClearSelected = clear_selected
        return select_items_option

    @staticmethod
    def convert_highlight_option(duration = 3, color = Color.Yellow):
        highlight_option = HighlightOption()
        highlight_option.HighlightTime = duration * 1000
        highlight_option.HighlightColor = _ConvertBaseTypeService.convert_color(color)
        return highlight_option
    
    @staticmethod
    def convert_save_image_option(img_width = 0, img_height = 0, xoffset = 0, yoffset  = 0):
        save_image_option = ScreenshotOption()
        save_image_option.ImgHeight = img_height
        save_image_option.ImgWidth = img_width
        save_image_option.XOffset = xoffset
        save_image_option.YOffset = yoffset
        return save_image_option

    @staticmethod
    def convert_open_browser_option(userdata_folder_mode, userdata_folder_path = ""):
        open_browser_option = dotnet_OpenBrowserOption()
        open_browser_option.UserDataFolderMode = _ConvertEnumService.convert_webuser_datafolder_mode_method(WebUserDataMode(userdata_folder_mode))
        open_browser_option.UserDataFolderPath = userdata_folder_path
        return open_browser_option

    @staticmethod
    def convert_event_data(type, properties):
        event_data = EventData()
        event_data.Type = _ConvertEnumService.convert_event_type(type)
        event_data.Properties = _ConvertBaseTypeService.convert_dict_str(properties)
        return event_data


class _ConvertEnumService:

    @staticmethod
    def convert_browser_type_enum(browser_type):
        if browser_type == BrowserType.Default:
            return dotnet_BrowserType.Default
        elif browser_type == BrowserType.IE:
            return dotnet_BrowserType.IE
        elif browser_type == BrowserType.Chrome:
            return dotnet_BrowserType.Chrome
        elif browser_type == BrowserType.FireFox:
            return dotnet_BrowserType.FireFox
        elif browser_type == BrowserType.Edge:
            return dotnet_BrowserType.Edge

    @staticmethod
    def convert_click_type_enum(click_type):
        if click_type == _InternalClickType.Click:
            return dotnet_ClickType.SingleClick
        elif click_type == _InternalClickType.DoubleClick:
            return dotnet_ClickType.DoubleClick
        elif click_type == _InternalClickType.Down:
            return dotnet_ClickType.DownClick
        elif click_type == _InternalClickType.Up:
            return dotnet_ClickType.UpClick

    @staticmethod
    def convert_mouse_button_enum(mouse_button):
        if mouse_button == MouseButton.Left:
            return dotnet_MouseButton.LeftButton
        elif mouse_button == MouseButton.Middle:
            return dotnet_MouseButton.MiddleButton
        elif mouse_button == MouseButton.Right:
            return dotnet_MouseButton.RightButton

    @staticmethod
    def convert_click_location_enum(click_location):
        if click_location == ClickLocation.Center:
            return dotnet_ClickLocation.Center
        elif click_location == ClickLocation.LeftTop:
            return dotnet_ClickLocation.LeftTop
        elif click_location == ClickLocation.LeftBottom:
            return dotnet_ClickLocation.LeftBottom
        elif click_location == ClickLocation.RightTop:
            return dotnet_ClickLocation.RightTop
        elif click_location == ClickLocation.RightBottom:
            return dotnet_ClickLocation.RightBottom

    @staticmethod
    def convert_click_method_enum(click_method):
        if click_method == ClickMethod.Default:
            return dotnet_ClickMethod.Default
        elif click_method == ClickMethod.MouseEmulation:
            return dotnet_ClickMethod.MouseEmulation
        elif click_method == ClickMethod.ControlInvocation:
            return dotnet_ClickMethod.ControlInvocation

    @staticmethod
    def convert_modifier_key_enum(modifier_key):
        if modifier_key == ModifierKey.NoneKey:
            return dotnet_ModifierKeysEnum.NoneKey
        elif modifier_key == ModifierKey.Alt:
            return dotnet_ModifierKeysEnum.Alt
        elif modifier_key == ModifierKey.Ctrl:
            return dotnet_ModifierKeysEnum.Ctrl
        elif modifier_key == ModifierKey.Shift:
            return dotnet_ModifierKeysEnum.Shift
        elif modifier_key == ModifierKey.Win:
            return dotnet_ModifierKeysEnum.Win

    @staticmethod
    def convert_window_mode_enum(window_mode):
        if window_mode == WindowMode.Auto:
            return dotnet_BringWindowType.Auto
        elif window_mode == WindowMode.TopMost:
            return dotnet_BringWindowType.TopMost
        elif window_mode == WindowMode.NoAction:
            return dotnet_BringWindowType.NoAction

    @staticmethod
    def convert_input_method(input_method):
        if input_method == InputMethod.Default:
            return dotnet_InputMethod.Default
        elif input_method == InputMethod.ControlSetValue:
            return dotnet_InputMethod.ControlSetValue
        elif input_method == InputMethod.KeyboardSimulateWithClick:
            return dotnet_InputMethod.KeyboardSimulationWithClick
        elif input_method == InputMethod.KeyboardSimulateWithSetFocus:
            return dotnet_InputMethod.KeyboardSimulationWithFocus

    @staticmethod
    def convert_clear_method(clear_method):
        if clear_method == ClearMethod.ControlClearValue:
            return dotnet_ClearMethod.ControlClearValue
        elif clear_method == ClearMethod.SendHotKey:
            return dotnet_ClearMethod.SendHotKey

    @staticmethod
    def convert_clear_hotkey_method(clear_hotkey):
        if clear_hotkey == ClearHotKey.CtrlA_Delete:
            return dotnet_ClearHotKey.CtrlA_Delete
        elif clear_hotkey == ClearHotKey.Home_ShiftEnd_Delete:
            return dotnet_ClearHotKey.Home_ShiftEnd_Delete
        elif clear_hotkey == ClearHotKey.End_ShiftHome_Delete:
            return dotnet_ClearHotKey.End_ShiftHome_Delete

    @staticmethod
    def convert_preaction_method(preaction):
        if preaction == PreAction.SetFocus:
            return dotnet_PrefixAction.SetFocus
        elif preaction == PreAction.Click:
            return dotnet_PrefixAction.Click

    @staticmethod
    def convert_checktype_method(check_type):
        if check_type == CheckType.Check:
            return dotnet_CheckType.Check
        elif check_type == CheckType.UnCheck:
            return dotnet_CheckType.UnCheck
        elif check_type == CheckType.Toggle:
            return dotnet_CheckType.Toggle

    @staticmethod
    def convert_webuser_datafolder_mode_method(webuser_datafolder_mode):
        if webuser_datafolder_mode == WebUserDataMode.Automatic:
            return dotnet_WebUserDataFolderMode.Automatic
        elif webuser_datafolder_mode == WebUserDataMode.DefaultFolder:
            return dotnet_WebUserDataFolderMode.DefaultFolder
        elif webuser_datafolder_mode == WebUserDataMode.CustomFolder:
            return dotnet_WebUserDataFolderMode.CustomFolder

    @staticmethod
    def convert_event_type(type):
        if type == EventTypes.ApiCall:
            return dotnet_EventTypes.ApiCall
        elif type == EventTypes.ExceptionReport:
            return dotnet_EventTypes.ExceptionReport

        
class _ConvertDotnetEnumService:

    @staticmethod
    def convert_automatiom_tech_method(automation):
        if(automation == dotnet_AutomationTech.Uia):
            return AutomationTech.Uia
        elif(automation == dotnet_AutomationTech.Java):
            return AutomationTech.Java
        elif(automation == dotnet_AutomationTech.IE):
            return AutomationTech.IE
        elif(automation == dotnet_AutomationTech.Chrome):
            return AutomationTech.Chrome
        elif(automation == dotnet_AutomationTech.Firefox):
            return AutomationTech.Firefox
        elif(automation == dotnet_AutomationTech.Sap):
            return AutomationTech.Sap
        elif(automation == dotnet_AutomationTech.Edge):
            return AutomationTech.Edge
        elif(automation == dotnet_AutomationTech.IA):
            return AutomationTech.IA

    @staticmethod
    def convert_browser_type_method(browser_type):
        if(browser_type == dotnet_BrowserType.Default):
            return BrowserType.Default
        elif(browser_type == dotnet_BrowserType.IE):
            return BrowserType.IE
        elif(browser_type == dotnet_BrowserType.Chrome):
            return BrowserType.Chrome
        elif(browser_type == dotnet_BrowserType.FireFox):
            return BrowserType.FireFox
        elif(browser_type == dotnet_BrowserType.Edge):
            return BrowserType.Edge