# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ez_parallel']

package_data = \
{'': ['*']}

install_requires = \
['click>=8,<8.1', 'coverage>=6.0b1', 'rich>=9.8.2,<10.0.0', 'typer>=0.4,<1.0']

extras_require = \
{':python_version < "3.8"': ['importlib_metadata>=1.6.0,<2.0.0']}

entry_points = \
{'console_scripts': ['ez-parallel = ez_parallel.__main__:app']}

setup_kwargs = {
    'name': 'ez-parallel',
    'version': '0.1.11',
    'description': 'Easy Parallel Multiprocessing',
    'long_description': '# ez-parallel\n\n<div align="center">\n\n[![Build status](https://github.com/j-rossi-nl/ez-parallel/workflows/build/badge.svg?branch=master&event=push)](https://github.com/j-rossi-nl/ez-parallel/actions?query=workflow%3Abuild)\n[![Python Version](https://img.shields.io/pypi/pyversions/ez-parallel.svg)](https://pypi.org/project/ez-parallel/)\n[![Dependencies Status](https://img.shields.io/badge/dependencies-up%20to%20date-brightgreen.svg)](https://github.com/j-rossi-nl/ez-parallel/pulls?utf8=%E2%9C%93&q=is%3Apr%20author%3Aapp%2Fdependabot)\n[![codecov](https://codecov.io/gh/j-rossi-nl/ez-parallel/branch/master/graph/badge.svg?token=VJE8DX0BH3)](https://codecov.io/gh/j-rossi-nl/ez-parallel)\n\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Security: bandit](https://img.shields.io/badge/security-bandit-green.svg)](https://github.com/PyCQA/bandit)\n[![Pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/j-rossi-nl/ez-parallel/blob/master/.pre-commit-config.yaml)\n[![Semantic Versions](https://img.shields.io/badge/%F0%9F%9A%80-semantic%20versions-informational.svg)](https://github.com/j-rossi-nl/ez-parallel/releases)\n[![License](https://img.shields.io/github/license/j-rossi-nl/ez-parallel)](https://github.com/j-rossi-nl/ez-parallel/blob/master/LICENSE)\n\nEasy Parallel Multiprocessing\n\n</div>\n\n## Installation\n\nWith `pip` or `pip3`:\n```bash\npip install -U ez-parallel\n```\nor\n```bash\npip install ez-parallel\n```\n\nWith `Poetry`:\n```bash\npoetry add ez-parallel\n```\n\n## Usage\n\n* Process a list of items by using parallel workers\n  * Define what a worker does\n  * Define how to iterate through the data\n  * Just run\n* Display a global progress bar\n* Does the same for multithread\n\n\n### Multithread vs Multiprocessing\n\nIn multiprocessing, new processes will be launched, they won\'t share memory. The user should implement a way \nto store the results of a worker and gather these results when `multiprocess()` returns.\n\nWith multithreading, new threads will be launched, they all share the memory of the parent process. This \nalso restricts the runtime to a single CPU-core, as threads from a process do not get allocated to different \ncores. There will be no performance improvement when the distributed work is CPU-bound. \n\nHow to choose? (guidelines)\n* CPU-heavy (data transformation, data preprocessing): multiprocessing. \n* IO-heavy (DB requests, File I/O): multithreading.  \n\n## Examples\n\n### How to process a list? \n\n```python\nimport time\n\nfrom ez_parallel import list_iterator, queue_worker, multiprocess\n\n@queue_worker\ndef work_one_thing(x: int) -> int:\n  # do something\n  a = x + 2\n  time.sleep(0.1)\n  \n  # Worked on ONE thing = return 1\n  return 1\n\n# Data\nthings_to_process = list(range(1000000))\n\n# Create the iterator over the things to process\niter_fn, nb_things = list_iterator(things_to_process)\n\n# Process all the things in parallel with 20 processes\nmultiprocess(\n  worker_fn=work_one_thing,\n  input_iterator_fn=iter_fn,\n  total=nb_things,\n  nb_workers=20,\n  description=\'Process the things\'\n)\n\n```\n\n### How to Process a list by batch?\n\n```python\nimport time\nfrom typing import List\n\nfrom ez_parallel import batch_iterator_from_sliceable, queue_worker, multiprocess\n\n\n@queue_worker\ndef work_one_thing(x: List[int]) -> int:\n  # do something\n  a = [y + 2 for y in x]\n  time.sleep(0.1)\n  \n  # Worked on ONE thing = return 1\n  return len(x)\n\n# Data\nthings_to_process = list(range(1000000))\n\n# Create the iterator over the things to process\n# This will yield batches of 128 things\niter_fn = batch_iterator_from_sliceable(items=things_to_process, batch_size=128)\nnb_things = len(things_to_process)\n\n# Process all the things in parallel with 20 processes\nmultiprocess(\n  worker_fn=work_one_thing,\n  input_iterator_fn=iter_fn,\n  total=nb_things,\n  nb_workers=20,\n  description=\'Process the things\'\n)\n\n```\n\n### How to Process a list by batch in multithread?\n\n```python\nimport time\nfrom typing import List\n\nfrom ez_parallel import batch_iterator_from_sliceable, queue_worker, multithread\n\n\n@queue_worker\ndef work_one_thing(x: List[int]) -> int:\n  # do something\n  a = [y + 2 for y in x]\n  time.sleep(0.1)\n  \n  # Worked on ONE thing = return 1\n  return len(x)\n\n# Data\nthings_to_process = list(range(1000000))\n\n# Create the iterator over the things to process\n# This will yield batches of 128 things\niter_fn = batch_iterator_from_sliceable(items=things_to_process, batch_size=128)\nnb_things = len(things_to_process)\n\n# Process all the things in parallel with 20 processes\nmultithread(\n  worker_fn=work_one_thing,\n  input_iterator_fn=iter_fn,\n  total=nb_things,\n  nb_workers=20,\n  description=\'Process the things\'\n)\n\n```\n\n### How to collect results in multiprocessing?\n\n(Suggestion using temporary files)\nIn this scenario, results are recorded as JSONL files, the final result is the concatenation of all files.\n\n```python\nimport glob\nimport json\nimport os\nimport random\nimport shutil\nimport string\nimport tempfile\nfrom typing import List\n\nfrom ez_parallel import batch_iterator_from_sliceable, queue_worker, multithread\n\n\ndef random_file_name() -> str:\n  return \'\'.join(random.choices(string.ascii_letters, k=32))  \n\n\n# All processes write in the same file\n# The OS will deal with concurrent access\ntmp_file = os.path.join(tempfile.gettempdir(), random_file_name())\n\n@queue_worker\ndef work_one_thing(x: List[int]) -> int:\n  # This call is blocking until the file can be written\n  with open(tmp_file, \'a\') as out:\n    for number in x:\n      out.write(json.dumps({"number": number, "square": number ** 2}) + \'\\n\')\n  \n  # Worked on ONE thing = return 1\n  return len(x)\n\n# Data\nthings_to_process = list(range(1000000))\n\n# Create the iterator over the things to process\n# This will yield batches of 128 things\niter_fn = batch_iterator_from_sliceable(items=things_to_process, batch_size=128)\nnb_things = len(things_to_process)\n\n# Process all the things in parallel with 20 processes\nmultithread(\n  worker_fn=work_one_thing,\n  input_iterator_fn=iter_fn,\n  total=nb_things,\n  nb_workers=20,\n  description=\'Process the things\'\n)\n\n# Collect all the data\nwith open(tmp_file, \'r\') as src:\n  data = [json.loads(line) for line in src] \n\n# Delete temporary file\nos.remove(tmp_file)      \n\n```\n\n### How to collect results in multithreading\n\nA lot easier and straightforward, because all the threads share the same memory.\n\n```python\nfrom typing import List\n\nfrom ez_parallel import batch_iterator, queue_worker, multithread\n\n# List are threadsafe in Python\nresults = []\n\n@queue_worker\ndef work_one_thing(x: List[int]) -> int:\n  # do something\n  results.extend({"number": y, "square": y ** 2} for y in x)\n  \n  # Worked on ONE thing = return 1\n  return len(x)\n\n# Data\nthings_to_process = list(range(1000000))\n\n# Create the iterator over the things to process\n# This will yield batches of 128 things\niter_fn, nb_things = batch_iterator(items=things_to_process, batch_size=128)\n\n# Process all the things in parallel with 20 processes\nmultithread(\n  worker_fn=work_one_thing,\n  input_iterator_fn=iter_fn,\n  total=nb_things,\n  nb_workers=20,\n  description=\'Process the things\'\n)\n\nprint(len(results))\n\n```\n\n\n\n\n## 🛡 License\n\n[![License](https://img.shields.io/github/license/j-rossi-nl/ez-parallel)](https://github.com/j-rossi-nl/ez-parallel/blob/master/LICENSE)\n\nThis project is licensed under the terms of the `MIT` license. See [LICENSE](https://github.com/j-rossi-nl/ez-parallel/blob/master/LICENSE) for more details.\n\n## 📃 Citation\n\n```\n@misc{ez-parallel,\n  author = {Julien Rossi},\n  title = {Easy Parallel Multiprocessing},\n  year = {2021},\n  publisher = {GitHub},\n  journal = {GitHub repository},\n  howpublished = {\\url{https://github.com/j-rossi-nl/ez-parallel}}\n}\n```\n\n## Credits\n\nThis project was generated with [`python-package-template`](https://github.com/TezRomacH/python-package-template).\n',
    'author': 'Julien Rossi',
    'author_email': 'j.rossi@uva.nl',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/j-rossi-nl/ez-parallel',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
