"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const have_resource_1 = require("@aws-cdk/assert/lib/assertions/have-resource");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: true,
                InviteMessageTemplate: have_resource_1.ABSENT,
            },
            EmailVerificationMessage: 'The verification code to your new account is {####}',
            EmailVerificationSubject: 'Verify your new account',
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_CODE',
                EmailMessage: 'The verification code to your new account is {####}',
                EmailSubject: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
            SmsConfiguration: {
                SnsCallerArn: {
                    'Fn::GetAtt': ['PoolsmsRoleC3352CE6', 'Arn'],
                },
                ExternalId: 'Pool',
            },
            lambdaTriggers: have_resource_1.ABSENT,
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Condition: {
                            StringEquals: {
                                'sts:ExternalId': 'Pool',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cognito-idp.amazonaws.com',
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            Policies: [
                {
                    PolicyDocument: {
                        Statement: [
                            {
                                Action: 'sns:Publish',
                                Effect: 'Allow',
                                Resource: '*',
                            },
                        ],
                        Version: '2012-10-17',
                    },
                    PolicyName: 'sns-publish',
                },
            ],
        });
    });
    test('self sign up option is correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            selfSignUpEnabled: true,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: false,
            },
        });
    });
    test('email verification via link is configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.LINK,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailVerificationMessage: have_resource_1.ABSENT,
            EmailVerificationSubject: have_resource_1.ABSENT,
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_LINK',
                EmailMessageByLink: 'Verify your account by clicking on {##Verify Email##}',
                EmailSubjectByLink: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
        });
    }),
        test('email and sms verification messages are validated', () => {
            const stack = new core_1.Stack();
            expect(() => new lib_1.UserPool(stack, 'Pool1', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    emailBody: 'invalid email body',
                },
            })).toThrow(/Verification email body/);
            expect(() => new lib_1.UserPool(stack, 'Pool2', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    emailBody: 'valid email body {####}',
                },
            })).not.toThrow();
            expect(() => new lib_1.UserPool(stack, 'Pool3', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    smsMessage: 'invalid sms message',
                },
            })).toThrow(/SMS message/);
            expect(() => new lib_1.UserPool(stack, 'Pool4', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    smsMessage: 'invalid sms message {####}',
                },
            })).not.toThrow();
            expect(() => new lib_1.UserPool(stack, 'Pool5', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.LINK,
                    emailBody: 'invalid email body {####}',
                },
            })).toThrow(/Verification email body/);
            expect(() => new lib_1.UserPool(stack, 'Pool6', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.LINK,
                    emailBody: 'invalid email body {##Verify Email##}',
                },
            })).not.toThrow();
        });
    test('validation is skipped for email and sms messages when tokens', () => {
        const stack = new core_1.Stack();
        const parameter = new core_1.CfnParameter(stack, 'Parameter');
        expect(() => new lib_1.UserPool(stack, 'Pool1', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.CODE,
                emailBody: parameter.valueAsString,
            },
        })).not.toThrow();
        expect(() => new lib_1.UserPool(stack, 'Pool2', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.CODE,
                smsMessage: parameter.valueAsString,
            },
        })).not.toThrow();
    });
    test('user invitation messages are configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userInvitation: {
                emailBody: 'invitation email body',
                emailSubject: 'invitation email subject',
                smsMessage: 'invitation sms',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                InviteMessageTemplate: {
                    EmailMessage: 'invitation email body',
                    EmailSubject: 'invitation email subject',
                    SMSMessage: 'invitation sms',
                },
            },
        });
    });
    test('smsRole property is recognized', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const role = aws_iam_1.Role.fromRoleArn(stack, 'smsRole', 'arn:aws:iam::664773442901:role/sms-role');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            smsRole: role,
            smsRoleExternalId: 'test-external-id',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            SmsConfiguration: {
                ExternalId: 'test-external-id',
                SnsCallerArn: role.roleArn,
            },
        });
    });
    test('import using id', () => {
        // GIVEN
        const stack = new core_1.Stack(undefined, undefined, {
            env: { region: 'some-region-1', account: '0123456789012' },
        });
        const userPoolId = 'test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolId(stack, 'userpool', userPoolId);
        expect(pool.userPoolId).toEqual(userPoolId);
        expect(pool.userPoolArn).toMatch(/cognito-idp:some-region-1:0123456789012:userpool\/test-user-pool/);
    });
    test('import using arn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userPoolArn = 'arn:aws:cognito-idp:us-east-1:0123456789012:userpool/test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolArn(stack, 'userpool', userPoolArn);
        expect(pool.userPoolId).toEqual('test-user-pool');
        expect(stack.resolve(pool.userPoolArn)).toEqual({
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':cognito-idp:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':userpool/test-user-pool',
                ]],
        });
    });
    test('support tags', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'myPool',
        });
        core_1.Tag.add(pool, 'PoolTag', 'PoolParty');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'myPool',
            UserPoolTags: {
                PoolTag: 'PoolParty',
            },
        });
    });
    test('lambda triggers via properties are correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = fooFunction(stack, 'preSignUp');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                PreSignUp: stack.resolve(fn.functionArn),
            },
        });
        expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: stack.resolve(fn.functionArn),
            Principal: 'cognito-idp.amazonaws.com',
        });
    });
    test('add* API correctly appends triggers', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const createAuthChallenge = fooFunction(stack, 'createAuthChallenge');
        const customMessage = fooFunction(stack, 'customMessage');
        const defineAuthChallenge = fooFunction(stack, 'defineAuthChallenge');
        const postAuthentication = fooFunction(stack, 'postAuthentication');
        const postConfirmation = fooFunction(stack, 'postConfirmation');
        const preAuthentication = fooFunction(stack, 'preAuthentication');
        const preSignUp = fooFunction(stack, 'preSignUp');
        const preTokenGeneration = fooFunction(stack, 'preTokenGeneration');
        const userMigration = fooFunction(stack, 'userMigration');
        const verifyAuthChallengeResponse = fooFunction(stack, 'verifyAuthChallengeResponse');
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool');
        pool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, createAuthChallenge);
        pool.addTrigger(lib_1.UserPoolOperation.CUSTOM_MESSAGE, customMessage);
        pool.addTrigger(lib_1.UserPoolOperation.DEFINE_AUTH_CHALLENGE, defineAuthChallenge);
        pool.addTrigger(lib_1.UserPoolOperation.POST_AUTHENTICATION, postAuthentication);
        pool.addTrigger(lib_1.UserPoolOperation.POST_CONFIRMATION, postConfirmation);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_AUTHENTICATION, preAuthentication);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_SIGN_UP, preSignUp);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_TOKEN_GENERATION, preTokenGeneration);
        pool.addTrigger(lib_1.UserPoolOperation.USER_MIGRATION, userMigration);
        pool.addTrigger(lib_1.UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE, verifyAuthChallengeResponse);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                CreateAuthChallenge: stack.resolve(createAuthChallenge.functionArn),
                CustomMessage: stack.resolve(customMessage.functionArn),
                DefineAuthChallenge: stack.resolve(defineAuthChallenge.functionArn),
                PostAuthentication: stack.resolve(postAuthentication.functionArn),
                PostConfirmation: stack.resolve(postConfirmation.functionArn),
                PreAuthentication: stack.resolve(preAuthentication.functionArn),
                PreSignUp: stack.resolve(preSignUp.functionArn),
                PreTokenGeneration: stack.resolve(preTokenGeneration.functionArn),
                UserMigration: stack.resolve(userMigration.functionArn),
                VerifyAuthChallengeResponse: stack.resolve(verifyAuthChallengeResponse.functionArn),
            },
        });
        [createAuthChallenge, customMessage, defineAuthChallenge, postAuthentication,
            postConfirmation, preAuthentication, preSignUp, preTokenGeneration, userMigration,
            verifyAuthChallengeResponse].forEach((fn) => {
            expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
                Action: 'lambda:InvokeFunction',
                FunctionName: stack.resolve(fn.functionArn),
                Principal: 'cognito-idp.amazonaws.com',
            });
        });
    });
    test('fails when the same trigger is added twice', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userpool = new lib_1.UserPool(stack, 'Pool');
        const fn1 = new lambda.Function(stack, 'fn1', {
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
        });
        const fn2 = new lambda.Function(stack, 'fn2', {
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
        });
        // WHEN
        userpool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, fn1);
        expect(() => userpool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, fn2)).toThrow(/already exists/);
    });
    test('no username aliases specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('fails when preferredUsername is used without username', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { preferredUsername: true },
        })).toThrow(/username/);
    });
    test('username and email are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true, email: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: ['email'],
        });
    });
    test('email and phone number are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: ['email', 'phone_number'],
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('email and phone number are autoverified, by default, if they are specified as signIn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            signInAliases: { email: true },
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            signInAliases: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            AutoVerifiedAttributes: ['email'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('explicit autoverify are correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true },
            autoVerify: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('sign in case sensitive is correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInCaseSensitive: false,
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameConfiguration: {
                CaseSensitive: false,
            },
        });
    });
    test('sign in case sensitive is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {});
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameConfiguration: have_resource_1.ABSENT,
        });
    });
    test('standard attributes default to mutable', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            standardAttributes: {
                fullname: {
                    required: true,
                },
                timezone: {
                    required: true,
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'name',
                    Required: true,
                    Mutable: true,
                },
                {
                    Name: 'zoneinfo',
                    Required: true,
                    Mutable: true,
                },
            ],
        });
    });
    test('mutable standard attributes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'Pool',
            standardAttributes: {
                fullname: {
                    required: true,
                    mutable: true,
                },
                timezone: {
                    required: true,
                    mutable: true,
                },
            },
        });
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            standardAttributes: {
                fullname: {
                    mutable: false,
                },
                timezone: {
                    mutable: false,
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool',
            Schema: [
                {
                    Mutable: true,
                    Name: 'name',
                    Required: true,
                },
                {
                    Mutable: true,
                    Name: 'zoneinfo',
                    Required: true,
                },
            ],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            Schema: [
                {
                    Name: 'name',
                    Required: false,
                    Mutable: false,
                },
                {
                    Name: 'zoneinfo',
                    Required: false,
                    Mutable: false,
                },
            ],
        });
    });
    test('schema is absent when attributes are not specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', { userPoolName: 'Pool' });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool',
            Schema: have_resource_1.ABSENT,
        });
    });
    test('optional mutable standardAttributes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'Pool',
            standardAttributes: {
                timezone: {
                    mutable: true,
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool',
            Schema: [
                {
                    Mutable: true,
                    Required: false,
                    Name: 'zoneinfo',
                },
            ],
        });
    });
    test('custom attributes with default constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute(),
                'custom-number-attr': new lib_1.NumberAttribute(),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'custom-string-attr',
                    AttributeDataType: 'String',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                },
                {
                    Name: 'custom-number-attr',
                    AttributeDataType: 'Number',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                },
            ],
        });
    });
    test('custom attributes with constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute({ minLen: 5, maxLen: 50 }),
                'custom-number-attr': new lib_1.NumberAttribute({ min: 500, max: 2000 }),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    AttributeDataType: 'String',
                    Name: 'custom-string-attr',
                    StringAttributeConstraints: {
                        MaxLength: '50',
                        MinLength: '5',
                    },
                },
                {
                    AttributeDataType: 'Number',
                    Name: 'custom-number-attr',
                    NumberAttributeConstraints: {
                        MaxValue: '2000',
                        MinValue: '500',
                    },
                },
            ],
        });
    });
    test('mfaTypes is ignored when mfaEnforcement is undefined or set to OFF', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.OFF,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: have_resource_1.ABSENT,
            EnabledMfas: have_resource_1.ABSENT,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'OFF',
            EnabledMfas: have_resource_1.ABSENT,
        });
    });
    test('sms mfa type is the default when mfaEnforcement is set to REQUIRED or OPTIONAL', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfa: lib_1.Mfa.OPTIONAL,
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.REQUIRED,
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: 'OPTIONAL',
            EnabledMfas: ['SMS_MFA'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'ON',
            EnabledMfas: ['SMS_MFA'],
        });
    });
    test('mfa type is correctly picked up when specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            mfa: lib_1.Mfa.REQUIRED,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EnabledMfas: ['SMS_MFA', 'SOFTWARE_TOKEN_MFA'],
        });
    });
    test('password policy is correctly set', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(2),
                minLength: 15,
                requireDigits: true,
                requireLowercase: true,
                requireUppercase: true,
                requireSymbols: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Policies: {
                PasswordPolicy: {
                    TemporaryPasswordValidityDays: 2,
                    MinimumLength: 15,
                    RequireLowercase: true,
                    RequireUppercase: true,
                    RequireNumbers: true,
                    RequireSymbols: true,
                },
            },
        });
    });
    test('password minimum length is set to the default when other parts of the policy is configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(2),
                requireDigits: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Policies: {
                PasswordPolicy: {
                    MinimumLength: 8,
                },
            },
        });
    });
    test('throws when tempPassword validity is not in round days', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.hours(30),
            },
        })).toThrow();
    });
    test('temp password throws an error when above the max', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(400),
            },
        })).toThrow(/tempPasswordValidity cannot be greater than/);
    });
    test('throws when minLength is out of range', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool1', {
            passwordPolicy: {
                minLength: 5,
            },
        })).toThrow(/minLength for password must be between/);
        expect(() => new lib_1.UserPool(stack, 'Pool2', {
            passwordPolicy: {
                minLength: 100,
            },
        })).toThrow(/minLength for password must be between/);
    });
    test('email transmission settings are recognized correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            emailSettings: {
                from: 'from@myawesomeapp.com',
                replyTo: 'replyTo@myawesomeapp.com',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailConfiguration: {
                From: 'from@myawesomeapp.com',
                ReplyToEmailAddress: 'replyTo@myawesomeapp.com',
            },
        });
    });
});
test('addClient', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    const userpool = new lib_1.UserPool(stack, 'Pool');
    userpool.addClient('UserPoolClient', {
        userPoolClientName: 'userpoolclient',
    });
    const imported = lib_1.UserPool.fromUserPoolId(stack, 'imported', 'imported-userpool-id');
    imported.addClient('UserPoolImportedClient', {
        userPoolClientName: 'userpoolimportedclient',
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
        ClientName: 'userpoolclient',
        UserPoolId: stack.resolve(userpool.userPoolId),
    });
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
        ClientName: 'userpoolimportedclient',
        UserPoolId: stack.resolve(imported.userPoolId),
    });
});
test('addDomain', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    const userpool = new lib_1.UserPool(stack, 'Pool');
    userpool.addDomain('UserPoolDomain', {
        cognitoDomain: {
            domainPrefix: 'userpooldomain',
        },
    });
    const imported = lib_1.UserPool.fromUserPoolId(stack, 'imported', 'imported-userpool-id');
    imported.addDomain('UserPoolImportedDomain', {
        cognitoDomain: {
            domainPrefix: 'userpoolimporteddomain',
        },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolDomain', {
        Domain: 'userpooldomain',
        UserPoolId: stack.resolve(userpool.userPoolId),
    });
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolDomain', {
        Domain: 'userpoolimporteddomain',
        UserPoolId: stack.resolve(imported.userPoolId),
    });
});
test('registered identity providers', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const userPool = new lib_1.UserPool(stack, 'pool');
    const provider1 = lib_1.UserPoolIdentityProvider.fromProviderName(stack, 'provider1', 'provider1');
    const provider2 = lib_1.UserPoolIdentityProvider.fromProviderName(stack, 'provider2', 'provider2');
    // WHEN
    userPool.registerIdentityProvider(provider1);
    userPool.registerIdentityProvider(provider2);
    // THEN
    expect(userPool.identityProviders).toEqual([provider1, provider2]);
});
function fooFunction(scope, name) {
    return new lambda.Function(scope, name, {
        functionName: name,
        code: lambda.Code.inline('foo'),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
    });
}
//# sourceMappingURL=data:application/json;base64,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