"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_a = props.userInvitation) === null || _a === void 0 ? void 0 : _a.emailBody,
            emailSubject: (_b = props.userInvitation) === null || _b === void 0 ? void 0 : _b.emailSubject,
            smsMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: (_d = props.emailSettings) === null || _d === void 0 ? void 0 : _d.from,
                replyToEmailAddress: (_e = props.emailSettings) === null || _e === void 0 ? void 0 : _e.replyTo,
            }),
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends UserPoolBase {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        return UserPool.fromUserPoolId(scope, id, core_1.Stack.of(scope).parseArn(userPoolArn).resourceName);
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        this.triggers[operation.operationName] = fn.functionArn;
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn,
        });
    }
    verificationMessageConfiguration(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailStyle) !== null && _b !== void 0 ? _b : VerificationEmailStyle.CODE;
        const emailSubject = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailSubject) !== null && _d !== void 0 ? _d : 'Verify your new account';
        const smsMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage) !== null && _f !== void 0 ? _f : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailBody) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases) !== null && _a !== void 0 ? _a : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a;
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
            };
        }
        else {
            const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
            const smsRole = (_a = props.smsRole) !== null && _a !== void 0 ? _a : new aws_iam_1.Role(this, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                    conditions: {
                        StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                    },
                }),
                inlinePolicies: {
                    /*
                     * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                     * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                     * Seems like a case of being excessively strict.
                     */
                    'sns-publish': new aws_iam_1.PolicyDocument({
                        statements: [
                            new aws_iam_1.PolicyStatement({
                                actions: ['sns:Publish'],
                                resources: ['*'],
                            }),
                        ],
                    }),
                },
            });
            return {
                externalId: smsRoleExternalId,
                snsCallerArn: smsRole.roleArn,
            };
        }
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_b = props.passwordPolicy.minLength) !== null && _b !== void 0 ? _b : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.requireLowercase,
            requireUppercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireUppercase,
            requireNumbers: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireDigits,
            requireSymbols: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => {
                var _a, _b;
                return ({
                    name: StandardAttributeNames[attrName],
                    mutable: (_a = attr.mutable) !== null && _a !== void 0 ? _a : true,
                    required: (_b = attr.required) !== null && _b !== void 0 ? _b : false,
                });
            });
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
}
exports.UserPool = UserPool;
const StandardAttributeNames = {
    address: 'address',
    birthdate: 'birthdate',
    email: 'email',
    familyName: 'family_name',
    gender: 'gender',
    givenName: 'given_name',
    locale: 'locale',
    middleName: 'middle_name',
    fullname: 'name',
    nickname: 'nickname',
    phoneNumber: 'phone_number',
    profilePicture: 'picture',
    preferredUsername: 'preferred_username',
    profilePage: 'profile',
    timezone: 'zoneinfo',
    lastUpdateTime: 'updated_at',
    website: 'website',
};
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).reduce((acc, v) => acc && (v === undefined), true);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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